import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import avatarByUrl from '../actions/loadAvatarByUrl';
import setUrlData from '../actions/setUrlData';
import avatarByFile from '../actions/loadAvatarByFile';
import {ErrorMessage} from './error.jsx';
import AvatarImage from './image.jsx';
import {AvatarLoadControls} from './formControls.jsx';
import {avatarCropper} from '../helpers/avatarCropper';
import SimpleForm from '@blocks/morda/avatar/components/simpleForm';
import {setAvatarError} from '@blocks/morda/avatar/actions';

export default class EditAvatar extends PureComponent {
    constructor(props) {
        super(props);
        this.handleUrl = this.handleUrl.bind(this);
        this.checkField = this.checkField.bind(this);
        this.handleFile = this.handleFile.bind(this);
    }

    checkField(url) {
        if (url) {
            this.props.dispatch(setUrlData(url));
        } else {
            this.props.dispatch(avatarCropper.destroyCropImage());
        }
    }

    handleUrl(url) {
        this.props.dispatch(avatarByUrl(url));
    }

    handleFile(event) {
        const {dispatch} = this.props;

        if (window.File && window.FileReader) {
            const file = event.target.files[0];

            dispatch(avatarByFile(file));
            return;
        }

        dispatch(setAvatarError('error-internal'));
    }

    render() {
        const {avatarInfo, isPhone, isCropNotSupported, dispatch} = this.props;
        const errorType = avatarInfo && avatarInfo.error;
        const disabled = avatarInfo && avatarInfo.status === 'loading';
        const isPhoneWithNormalStatus = isPhone && !disabled && avatarInfo.status !== 'processing';
        const showAvatarLoadControls = isPhoneWithNormalStatus || !isPhone;
        const hasAvatarChanged = avatarInfo.hasChanged;

        if (isCropNotSupported) {
            return (
                <>
                    <ErrorMessage visible={errorType === 'error-internal'} errorType={errorType} />
                    <div className='avatar-block'>
                        <AvatarImage avatarInfo={avatarInfo} dispatch={dispatch} />
                        <SimpleForm />
                    </div>
                </>
            );
        }

        return (
            <div>
                <ErrorMessage visible={errorType === 'error-internal'} errorType={errorType} />

                <div className='avatar-block'>
                    <AvatarImage avatarInfo={avatarInfo} dispatch={dispatch} />
                    {showAvatarLoadControls && (
                        <AvatarLoadControls
                            dispatch={dispatch}
                            isByUrl={avatarInfo.isByUrl}
                            status={avatarInfo.status}
                            trackId={avatarInfo.track_id}
                            disabled={disabled}
                            errorType={errorType}
                            handleFile={this.handleFile}
                            handleUrl={this.handleUrl}
                            hasAvatarChanged={hasAvatarChanged}
                            checkField={this.checkField}
                        />
                    )}
                </div>
            </div>
        );
    }
}

EditAvatar.propTypes = {
    avatarInfo: PropTypes.object,
    dispatch: PropTypes.func,
    isPhone: PropTypes.bool,
    isCropNotSupported: PropTypes.bool.isRequired
};
