import cropperjs from 'cropperjs/dist/cropper.min.js';
import {setAvatarError, updateAvatar, updateAvatarStatus} from '@blocks/morda/avatar/actions';
import dataURItoBlob from '@blocks/morda/avatar/helpers/dataUriToBlob';
import updateAvatarImage from '@blocks/morda/avatar/actions/updateAvatarImage';
import checkRetpath from '@blocks/morda/avatar/actions/checkRetpath';
import processErrors from '@blocks/morda/avatar/helpers/processErrors';
import forceClearFileInput from '@blocks/morda/avatar/helpers/forceClearFileInput';
import {setEditMode} from '@blocks/common/actions';
import api from '@blocks/api';

export const avatarCropper = {
    init() {
        const image = document.getElementById('user-avatar');
        const Cropper = cropperjs;

        this.cropper = new Cropper(image, {
            aspectRatio: 1 / 1, // https://github.com/fengyuanchen/cropperjs#aspectratio
            checkCrossOrigin: false,
            guides: false,
            center: false,
            background: false,
            zoomable: false
        });

        image.addEventListener('crop', this.setMinCropperWidth);
        image.addEventListener('cropend', this.setMinCropperWidth);
    },

    setMinCropperWidth() {
        const frameParams = this.cropper.getData(true);
        const frameWidth = frameParams.width;
        const minWidth = 200;

        if (frameWidth < minWidth) {
            this.cropper.setData({width: minWidth});
        }
    },

    cropAvatar() {
        return (dispatch, getState) => {
            const avatarState = getState().changeAvatar;
            const result = this.cropper.getCroppedCanvas({fillColor: 'white'});

            if (result) {
                const dataURL = result.toDataURL(`image/${avatarState.type}`, avatarState.compression);

                dispatch(this.loadCroppedAvatar(dataURL, avatarState.track_id));
                return;
            }

            dispatch(setAvatarError('error-internal'));
        };
    },

    loadCroppedAvatar(croppedData, track) {
        return (dispatch, getState) => {
            const formData = new FormData();
            const file = dataURItoBlob(croppedData);
            const state = getState();
            const {
                common: {csrf}
            } = state;

            formData.append('file', file, 'avatar.jpg');
            formData.append('consumer', 'passport');
            formData.append('track_id', track);
            formData.append('csrf_token', csrf);

            $.ajax({
                url: '/profile/avatars/load/',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json'
            })
                .done((response) => {
                    const {body} = response;

                    if (body && body.status === 'ok') {
                        dispatch(this.updateAvatarDataAfterCrop(body.avatar_url));
                        dispatch(updateAvatarImage(body.avatar_url));
                        dispatch(checkRetpath());

                        api.writeStatbox({
                            mode: 'profile_change_avatar',
                            track_id: track,
                            action: 'update'
                        });
                    } else {
                        dispatch(this.destroyCropImage(processErrors(response)));
                    }
                })
                .fail(() => {
                    dispatch(this.destroyCropImage('error-internal'));
                })
                .always(() => {
                    forceClearFileInput();
                });
        };
    },

    destroyCropper() {
        if (this.cropper) {
            this.cropper.destroy();
        }
    },

    destroyCropImage(error = '') {
        return (dispatch, getState) => {
            const state = getState() || {};
            const {
                changeAvatar: {backupAvatar, isByUrl, status}
            } = state;
            const url = backupAvatar.url;
            const isLoadByUrl = isByUrl && status === 'processing';
            const data = {
                avatarUrl: url,
                isByUrl: isLoadByUrl,
                loadUrlPath: '',
                hasChanged: false
            };

            this.destroyCropper();
            dispatch(updateAvatarStatus('normal'));
            dispatch(updateAvatar(data));
            dispatch(setAvatarError(error));
        };
    },

    updateAvatarDataAfterCrop() {
        return (dispatch, getState) => {
            const state = getState();
            const {changeAvatar} = state;
            const {isByUrl} = changeAvatar;
            const data = {
                hasChanged: true,
                loadUrlPath: ''
            };

            if (isByUrl) {
                data.isByUrl = false;
            }

            avatarCropper.destroyCropper(null);
            dispatch(setEditMode(null));
            dispatch(updateAvatarStatus('normal'));
            dispatch(updateAvatar(data));
        };
    }
};
