import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {Modal} from '@components/Modal';
import {Link as LegoLink} from '@components/Link';
import {getCardsInfo, createBinding, EDIT_MODE_NAME, CARDS_LINK} from './';
import Card from './card.jsx';
import CardsControl from './cards_control.jsx';
import Link from '@blocks/morda/components/link/link.jsx';
import {setEditMode} from '@blocks/common/actions';
import metrics from '@blocks/metrics.js';
import {BILLING_METRICS_PREFIX, BILLING_GOAL_PREFIX} from '@blocks/morda/billing_info';
import {selectCard, setErrors} from '@blocks/morda/cards/actions';

export class Cards extends PureComponent {
    constructor(props) {
        super(props);

        this.displayName = 'Cards';
        this.showModal = this.showModal.bind(this);
        this.bindCard = this.bindCard.bind(this);
    }

    componentDidMount() {
        const {dispatch, billing} = this.props;
        const {cards = []} = billing;

        if (!cards.length) {
            dispatch(getCardsInfo());
        }
    }

    showModal() {
        const {dispatch} = this.props;

        metrics.send([BILLING_METRICS_PREFIX, 'Открыли попап']);
        dispatch(setEditMode(EDIT_MODE_NAME));
    }

    bindCard() {
        const {dispatch} = this.props;

        dispatch(setEditMode(EDIT_MODE_NAME));
        dispatch(createBinding());
    }

    sendMetrics(msg, goal) {
        metrics.send([BILLING_METRICS_PREFIX, msg]);
        metrics.goal(`${BILLING_GOAL_PREFIX}${goal}`);
    }

    closeModal = () => {
        const {dispatch} = this.props;

        dispatch(setErrors([]));
        dispatch(setEditMode(''));
        dispatch(selectCard(''));
    };

    render() {
        const {dispatch, billing, retpath, settings = {}, edit} = this.props;
        const {cards: _cards = []} = billing;
        const cards = _cards.filter(({proto}) => proto !== 'yamoney_walletapi');
        const firstThreeCards = cards.slice(0, 3);
        const {ua = {}} = settings;
        const {isMobile, isTouch, isTablet} = ua;
        const isPhone = (isMobile || isTouch) && !isTablet;

        return (
            <div>
                {Boolean(firstThreeCards.length) && (
                    <div>
                        <div
                            className={classnames('cards', {
                                'cards-two': firstThreeCards.length === 2,
                                'cards-many': firstThreeCards.length > 2
                            })}
                        >
                            {firstThreeCards.map((card, index) => (
                                <Card card={card} key={index} settings={settings} dispatch={dispatch} />
                            ))}
                        </div>

                        <div className='p-control-link'>
                            <span onMouseDown={() => this.sendMetrics('Редактировать', 'edit')}>
                                {isPhone ? (
                                    <Link url={CARDS_LINK}>{i18n('Frontend.common.edit')}</Link>
                                ) : (
                                    <LegoLink onClick={this.showModal} pseudo={true}>
                                        {i18n('Frontend.common.edit')}
                                    </LegoLink>
                                )}
                            </span>
                        </div>
                    </div>
                )}

                <div className='p-control-link'>
                    <span onMouseDown={() => this.sendMetrics('Привязать платёжную карту', 'bind_init_widget')}>
                        {isPhone ? (
                            <Link url={CARDS_LINK}>{i18n('Profile.cards.widget.title')}</Link>
                        ) : (
                            <LegoLink onClick={this.bindCard} pseudo={true}>
                                {i18n('Profile.cards.widget.title')}
                            </LegoLink>
                        )}
                    </span>
                </div>
                <div className='section-body-info section-body-info__bottom_shift__m'>
                    {i18n('Profile.cards.widget.info')}
                </div>

                {Boolean(!isPhone) && (
                    <Modal
                        visible={edit === 'cards'}
                        zIndexGroupLevel={-0.5}
                        onCloserClick={this.closeModal}
                        dataT='cards'
                    >
                        <CardsControl
                            retpath={retpath}
                            billing={billing}
                            dispatch={dispatch}
                            settings={settings}
                            modal={true}
                        />
                    </Modal>
                )}
            </div>
        );
    }
}

Cards.propTypes = {
    dispatch: PropTypes.func.isRequired,
    billing: PropTypes.shape({
        cards: PropTypes.array
    }),
    retpath: PropTypes.string,
    settings: PropTypes.shape({
        ua: PropTypes.shape({
            isMobile: PropTypes.bool,
            isTouch: PropTypes.bool,
            isTablet: PropTypes.bool
        })
    }),
    edit: PropTypes.string.isRequired
};
