import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {Spin} from '@components/Spin';
import {Button} from '@components/Button';
import {Modal} from '@components/Modal';
import Card from './card.jsx';
import ReturnToService from '@blocks/morda/components/return_button/return_button.jsx';
import {getCardsInfo, splitCardNumber, createBinding, unbindCard, BILLING_CARDS_LIMIT} from './';
import {selectCard} from './actions';
import metrics from '@blocks/metrics.js';
import {CardUnbind} from './card_unbind';
import {BILLING_METRICS_PREFIX, BILLING_GOAL_PREFIX} from '@blocks/morda/billing_info';
import {hasExp} from '@blocks/selectors';

const ERRORS = {
    internal: i18n('Profile.cards.errors.internal'),
    too_many_cards: i18n('Profile.cards.errors.too_many_cards')
};

export class CardsList extends Component {
    state = {isUnbindModalOpened: false, savedCardId: null};

    unSelectCard = () => {
        const {dispatch, billing} = this.props;
        const {selected = ''} = billing;

        if (selected !== '') {
            dispatch(selectCard(''));
        }
    };

    componentDidMount() {
        const {dispatch, modal, billing} = this.props;
        const {cards = []} = billing;

        if (!modal && cards.length === 0) {
            dispatch(getCardsInfo());
        }
    }

    componentWillUnmount() {
        const {dispatch} = this.props;

        dispatch(selectCard(''));
    }

    createBinding = () => {
        const {dispatch} = this.props;

        metrics.send([BILLING_METRICS_PREFIX, 'Привятать карту']);
        metrics.goal(`${BILLING_GOAL_PREFIX}bind_init`);
        dispatch(createBinding());
    };

    unbindCard = (id) => {
        const {dispatch} = this.props;

        metrics.send([BILLING_METRICS_PREFIX, 'Отвязать карту']);
        metrics.goal(`${BILLING_GOAL_PREFIX}unbind`);
        dispatch(unbindCard(id));
    };

    unbindCardOnClick = (id) => {
        const {isUnbindCardAlertExp} = this.props;

        if (isUnbindCardAlertExp) {
            metrics.send([BILLING_METRICS_PREFIX, 'Отвязать карту', 'Открытие']);
            this.setState({savedCardId: id, isUnbindModalOpened: true});
        } else {
            this.unbindCard(id);
        }
    };

    unbindSavedCard = () => {
        const {hasPlus} = this.props;
        const {savedCardId} = this.state;

        const isPlus = hasPlus && this.getRealCards().length === 1;

        metrics.send([BILLING_METRICS_PREFIX, 'Отвязать карту', 'Удалить', isPlus ? 'Плюс' : 'Любая']);

        this.unbindCard(savedCardId);
    };

    closeModal = () => this.setState({savedCardId: null, isUnbindModalOpened: false});

    getErrorsMsgs = (errors = []) => {
        return errors.map((code, index) => <div key={index}>{ERRORS[code] || ERRORS.internal}</div>);
    };

    isCardBindAvailable = () => {
        const {billing: {cards = []} = {}} = this.props;

        return cards.filter(({proto}) => ['trust', 'fake'].includes(proto)).length < BILLING_CARDS_LIMIT;
    };

    getRealCards = () => {
        const {billing: {cards = []} = {}, uid, hasRestorePhone} = this.props;

        return cards
            .filter((card) => {
                const {isFamilyCard, familyAdminUid, proto} = card;
                const isFamilyAdmin = isFamilyCard && familyAdminUid === uid;
                const isNotAdminAndCanSeeFamilyCard = isFamilyCard && !isFamilyAdmin && hasRestorePhone;

                return isFamilyCard ? isFamilyAdmin || isNotAdminAndCanSeeFamilyCard : proto !== 'yamoney_walletapi';
            })
            .map((card) => {
                const {isFamilyCard, familyAdminUid} = card;
                const isFamilyAdmin = isFamilyCard && familyAdminUid === uid;
                const canUnbind = isFamilyCard ? isFamilyAdmin : true;

                return {...card, canUnbind, isFamilyCard: isFamilyCard && !isFamilyAdmin};
            });
    };

    onModalSaveClick = () => metrics.send([BILLING_METRICS_PREFIX, 'Отвязать карту', 'Закрытие', 'Оставить']);
    onModalCloserClick = () => {
        metrics.send([BILLING_METRICS_PREFIX, 'Отвязать карту', 'Закрытие', 'Крестик']);
        this.closeModal();
    };
    onModalClose = () => {
        metrics.send([BILLING_METRICS_PREFIX, 'Отвязать карту', 'Закрытие', 'Фон']);
        this.closeModal();
    };

    render() {
        const {modal, billing = {}, settings = {}, common, retpath, isUnbindCardAlertExp} = this.props;
        const {selected = '', errors = [], bindProgress = false, progress = false, unbindCardId} = billing;
        const cards = this.getRealCards();
        const {yamoneyCards = '', tld = 'ru'} = settings;
        const {isMobile, isTouch, isTablet} = settings.ua;
        const isPhone = (isMobile || isTouch) && !isTablet;
        const yamoneyCardsURL = yamoneyCards.replace('%tld%', tld);
        const isCardBindAvailable = this.isCardBindAvailable();
        const {isUnbindModalOpened} = this.state;
        const showAlert = hasExp({common}, 'show-cards-list-alert');

        return (
            <div data-t='cards-list'>
                <div className='section-head'>
                    <h2 className='section-title'>{i18n('Profile.cards.title')}</h2>
                </div>

                {Boolean(showAlert) && (
                    <div className='p-control-alert-block p-control-alert-block__internal'>
                        <div className='p-control-alert-block-msg'>{i18n('Profile.cards.alert')}</div>
                    </div>
                )}

                {Boolean(errors.length) && (
                    <div className='p-control-error-block p-control-error-block__internal'>
                        <div className='p-control-error-block-msg'>{this.getErrorsMsgs(errors)}</div>
                    </div>
                )}

                {cards.length ? (
                    <div data-t='cards-list:list'>
                        <div className='separator p-control-modal-separator p-control-modal-separator-no-offset-y' />
                        {cards.map((card, index) => (
                            <div
                                key={index}
                                onMouseOver={this.unSelectCard}
                                onTouchStart={this.unSelectCard}
                                className={`cards-list-item ${selected === card.id ? 'cards-list-item-selected' : ''}`}
                            >
                                <div className='cards-list-item-control cards-list-item-info-wrap'>
                                    <div className='cards-list-item-control'>
                                        <Card card={card} isStatic={true} settings={settings} />
                                    </div>
                                    <div className='cards-list-item-control cards-list-item-info'>
                                        <div>{card.system}</div>
                                        <div>
                                            {card.isFamilyCard
                                                ? i18n('Profile2.bank_cards.family')
                                                : card.number &&
                                                  splitCardNumber(
                                                      card.number.length === 14
                                                          ? card.number.replace('****', '******')
                                                          : card.number
                                                  )}
                                        </div>
                                        <div className='cards-list-item-info-name'>{card.name}</div>
                                    </div>
                                </div>
                                {card.canUnbind && (
                                    <div className='cards-list-item-control cards-list-item-unbind'>
                                        {card.proto === 'yamoney_walletapi' && (
                                            <Button
                                                size='l'
                                                view='pseudo'
                                                type='link'
                                                url={yamoneyCardsURL}
                                                text={i18n('Profile.cards.gotoyamoney')}
                                                width={isMobile ? 'max' : 'auto'}
                                            />
                                        )}
                                        {(card.proto === 'trust' || card.proto === 'fake') && (
                                            <Button
                                                view='pseudo'
                                                size='l'
                                                type='link'
                                                text={i18n('Profile.cards.unbind')}
                                                disabled={unbindCardId === card.id}
                                                onClick={() => this.unbindCardOnClick(card.id)}
                                                width={isMobile ? 'max' : 'auto'}
                                            />
                                        )}
                                    </div>
                                )}
                            </div>
                        ))}
                        <div className='separator p-control-modal-separator p-control-modal-separator-no-offset-y' />
                    </div>
                ) : (
                    <div className='p-control-modal-message p-control-modal-message__hint'>
                        {i18n('Profile.cards.info')}
                    </div>
                )}

                <div className='cards-list-spin'>
                    <Spin progress={progress} size='s' />
                </div>

                <div className='p-control-saveblock'>
                    <div className='p-control-saveblock-cell-left'>
                        <ReturnToService retpath={retpath} isModal={modal} isMobile={isPhone} />
                    </div>
                    <div className='p-control-saveblock-cell-right personal-saveblock'>
                        <span className='p-control-saveblock-button'>
                            <Button
                                size='l'
                                text={i18n('Profile.cards.bind')}
                                view='action'
                                type='link'
                                disabled={!isCardBindAvailable || bindProgress}
                                onClick={this.createBinding}
                                width={isMobile ? 'max' : 'auto'}
                            />
                        </span>
                        <span className='personal-spinner'>
                            <Spin size='s' progress={bindProgress} />
                        </span>
                    </div>
                </div>
                {isUnbindCardAlertExp && (
                    <Modal
                        visible={isUnbindModalOpened}
                        dataT='unbind-card-modal'
                        onClose={this.onModalClose}
                        onCloserClick={this.onModalCloserClick}
                    >
                        <CardUnbind
                            closeModal={this.closeModal}
                            onSaveClick={this.onModalSaveClick}
                            onUnbindClick={this.unbindSavedCard}
                            isOneBindedCard={cards.length === 1}
                        />
                    </Modal>
                )}
            </div>
        );
    }
}

CardsList.propTypes = {
    modal: PropTypes.bool,
    dispatch: PropTypes.func.isRequired,
    billing: PropTypes.shape({
        cards: PropTypes.array,
        selected: PropTypes.string,
        progress: PropTypes.bool,
        errors: PropTypes.array,
        isUpdated: PropTypes.bool
    }),
    isUnbindCardAlertExp: PropTypes.bool.isRequired,
    hasPlus: PropTypes.bool.isRequired,
    retpath: PropTypes.string,
    common: PropTypes.object,
    settings: PropTypes.shape({
        ua: PropTypes.shape({
            isMobile: PropTypes.bool,
            isTouch: PropTypes.bool,
            isTablet: PropTypes.bool
        })
    }),
    uid: PropTypes.number.isRequired,
    hasRestorePhone: PropTypes.bool.isRequired
};
