import api from '@blocks/api';
import {saveActionForRepeat} from '@blocks/common/actions';
import {
    setCardsInfo,
    setCardsInfoProgressState,
    setCardsBindProgressState,
    setErrors,
    setFrameVisibleState,
    setUnbindCard,
    setToken,
    setFrameData,
    setAbortBillingState
} from './actions';
import metrics from '@blocks/metrics.js';
import TrustSDK from '@yandex-trust-sdk/base';
import {BILLING_GOAL_PREFIX} from '@blocks/morda/billing_info';

export const EDIT_MODE_NAME = 'cards';
export const CARDS_LINK = '/profile/cards';
export const BILLING_CARDS_LIMIT = 5;
const ZERO = 0;
const CHUNK_LENGTH = 4;
const ERROR_TIMEOUT = 10000;

export function getCardsInfo() {
    return function(dispatch, getState) {
        const state = getState() || {};
        const {common = {}} = state;
        const {csrf} = common;

        dispatch(setCardsInfoProgressState(true));
        api.request('billing.cardsinfo', {csrf})
            .done((response) => {
                const {cards, status, errors} = response;

                if (status !== 'error') {
                    dispatch(setCardsInfo(cards));
                } else {
                    dispatch(setErrors(errors));
                }
            })
            .fail((response) => {
                dispatch(setCardsInfoProgressState(false));
                dispatch(setErrors(response.errors || ['internal']));
            });
    };
}

export function createBinding() {
    return function(dispatch, getState) {
        const {
            settings: {language, tld = 'ru', ua: {isMobile, isTouch} = {}},
            common: {csrf},
            billing: {lastClear}
        } = getState();

        dispatch(setErrors([]));
        dispatch(setCardsBindProgressState(true));

        dispatch(saveActionForRepeat(createBinding));
        api.request('billing.createbinding', {
            templateTag: isMobile || isTouch ? 'mobile/form' : 'desktop/form',
            language,
            domain_sfx: tld,
            csrf
        })
            .done((response = {}) => {
                if (getState().billing.lastClear !== lastClear) {
                    return;
                }

                const {token, status, errors} = response;

                if (status === 'error') {
                    dispatch(setErrors(errors || ['internal']));
                    return;
                }

                if (token) {
                    dispatch(setToken(token));
                    api.request('billing.dobinding', {
                        token,
                        csrf
                    })
                        .done((dobindingResponse = {}) => {
                            const {url, origin, errors, status} = dobindingResponse;

                            if (status === 'error') {
                                dispatch(setErrors(errors || ['internal']));
                                return;
                            }

                            dispatch(setFrameData({url, origin}));
                        })
                        .fail((failResponse) => {
                            const {errors} = failResponse;

                            dispatch(setErrors(errors || ['internal']));
                            dispatch(setCardsBindProgressState(false));
                        });

                    return;
                }

                dispatch(setErrors(['internal']));
                dispatch(setCardsBindProgressState(false));
            })
            .fail((response) => {
                if (getState().billing.lastClear !== lastClear) {
                    return;
                }

                const {errors} = response;

                if (!errors.includes('password.required')) {
                    dispatch(setErrors(errors || ['internal']));
                }

                dispatch(setCardsBindProgressState(false));
            });
    };
}

export function unbindCard(id) {
    return function(dispatch, getState) {
        const state = getState() || {};
        const {common = {}} = state;
        const {csrf} = common;

        dispatch(saveActionForRepeat(unbindCard, id));
        dispatch(setUnbindCard(id));

        api.request('billing.unbindcard', {
            id,
            csrf
        })
            .done((response = {}) => {
                const {status, errors} = response;

                if (status === 'error') {
                    dispatch(setErrors(errors || ['internal']));
                    return;
                }

                dispatch(getCardsInfo());
                dispatch(setUnbindCard(''));
            })
            .fail((response) => {
                const {errors} = response;

                if (!errors.includes('password.required')) {
                    dispatch(setErrors(errors || ['internal']));
                }

                dispatch(setUnbindCard(''));
            });
    };
}

export function splitCardNumber(number) {
    if (!number) {
        return '';
    }

    const numberLength = number.length;

    return [...Array(Math.ceil(numberLength / CHUNK_LENGTH))]
        .map((x, index) => {
            const stop = numberLength - index * CHUNK_LENGTH;
            const start = stop - CHUNK_LENGTH;

            return number.slice(start >= ZERO ? start : ZERO, stop);
        })
        .reverse()
        .join(' ');
}

export function resetCardBindingForm() {
    return function(dispatch) {
        dispatch(setFrameVisibleState(false));
        dispatch(setCardsBindProgressState(false));
        dispatch(setFrameData({url: '', origin: ''}));
    };
}

class Api {
    constructor(dispatch) {
        this.dispatch = dispatch;

        this.errorTimeout = setTimeout(this.onError, ERROR_TIMEOUT);
    }
    setListeners = (handler) => {
        if (window.addEventListener) {
            window.addEventListener('message', handler, false);
        } else {
            window.attachEvent('onmessage', handler);
        }
    };
    removeListeners = (handler) => {
        if (window.addEventListener) {
            window.removeEventListener('message', handler, false);
        } else {
            window.detachEvent('onmessage', handler);
        }
        clearTimeout(this.errorTimeout);
    };
    closeBilling = ({isAbort = false} = {}) => {
        const {dispatch} = this;

        dispatch(setAbortBillingState(isAbort));
        dispatch(resetCardBindingForm());
    };
    onComplete = () => {
        const {dispatch} = this;

        dispatch(getCardsInfo());
        this.closeBilling();
        metrics.goal(`${BILLING_GOAL_PREFIX}bind_complete`);
    };
    handleResponse(event) {
        const {dispatch} = this;

        let response = event.data || '';

        if (typeof response === 'string') {
            try {
                response = JSON.parse(event.data);
            } catch (error) {
                api.log(`Billing error with '${error}'`);
            }
        }

        if (
            (response && response.data && response.data.value === 'ready') ||
            response === 'payment:loaded' ||
            (response && response.type === 'form-ready')
        ) {
            clearTimeout(this.errorTimeout);
            dispatch(setFrameVisibleState(true));
            dispatch(setCardsBindProgressState(false));
            dispatch(setAbortBillingState(false));
        }

        if (
            (response && response.data && (response.data.value === 'success' || response.data === '3ds-success')) ||
            (response && response.type === 'payment-complete') ||
            (response && response.type === 'auth-end') ||
            (response && response.type === 'success')
        ) {
            this.onComplete();
        }

        if (
            (response && response.action === 'abort') ||
            (response && response.data && response.data.value === 'abort') ||
            response === '{"action":"abort"}' ||
            (response && response.type === 'abort')
        ) {
            this.closeBilling({isAbort: true});
        }

        if ((response && response.type === 'error') || (response && response.type === 'payment-fail')) {
            this.onError();
        }

        api.log(
            `Billing responded with event '${JSON.stringify(
                response.data || response.action || response.type || response
            )}'`
        );
    }
    onError = () => {
        const {dispatch} = this;

        dispatch(setErrors(['internal']));
        dispatch(setCardsBindProgressState(false));
        this.closeBilling();

        api.log('Billing timeout error');
    };
}

export class TrustApi extends Api {
    constructor(dispatch, lang, isTesting, selector, url) {
        super(dispatch);
        this.setListeners(this.onResponse);
        this.sdk = TrustSDK.create({lang, apiHost: `https://${isTesting ? 'trust-test' : 'trust'}.yandex.ru`});
        this.mount(selector, url);
    }

    mount = async (selector, url) => {
        this.bindFrame = await this.sdk.bindCard({url});
        this.bindFrame.mount(selector, {className: 'cards-binding-form cards-binding-form-trust'});
    };

    destroy = () => {
        this.bindFrame.unmount();
        this.removeListeners(this.onResponse);
    };

    onResponse = (event) => {
        if (this.bindFrame && event.origin === this.bindFrame.frameOrigin) {
            this.handleResponse(event);
        }
    };
}
