import React, {Component} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {connect} from 'react-redux';
import {Modal} from '@components/Modal';
import {Input} from '@components/Input';
import {Button} from '@components/Button';
import {Captcha} from '@components/Captcha';
import {validateForm, hideDomik} from './actions';
import {checkHuman} from './checkHuman';

const errorMessages = {
    password: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('Profile.domik.password-not-matched')
    },
    captcha: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('Profile.domik.captcha-not-matched')
    }
};

class Domik extends Component {
    constructor(props) {
        super(props);
        this.submitForm = this.submitForm.bind(this);
        this.hideDomik = this.hideDomik.bind(this);
        this.onHistoryChange = this.onHistoryChange.bind(this);
        this.handleInput = this.handleInput.bind(this);

        this.state = {
            password: '',
            captcha: '',
            errorIsShowed: false
        };
    }

    submitForm(e) {
        e.preventDefault();
        const data = this.getFieldValues();

        this.props.dispatch(validateForm(data));
        this.props.dispatch(checkHuman(data));
        this.setState({errorIsShowed: true});
    }

    getFieldValues() {
        const domik = this.props.domik;
        const data = {};

        if (domik.requestCaptcha) {
            data.captcha = this.state.captcha;
        }

        if (domik.requestPassword) {
            data.password = this.state.password;
        }

        return data;
    }

    handleInput(event) {
        const {target = {}} = event;
        const {value, name} = target;

        this.setState({
            [name]: value,
            errorIsShowed: false
        });
    }

    hideDomik() {
        this.props.dispatch(hideDomik());
    }

    onHistoryChange() {
        this.hideDomik();
    }

    componentDidMount() {
        const {requestPassword, requestCaptcha} = this.props.domik;

        if (requestPassword) {
            this.password.focus();
        } else if (requestCaptcha) {
            this.captcha.focus();
        }

        window.addEventListener('popstate', this.onHistoryChange);
    }

    componentWillUnmount() {
        window.removeEventListener('popstate', this.onHistoryChange);
    }

    render() {
        const props = this.props;
        const {isMobileCaptcha, captcha, lang, domik, is2faEnabled, isMobile} = props;
        const {requestPassword, requestCaptcha, errors} = domik;
        const domikLogoLang = ['ru', 'uk'].indexOf(lang) > -1 ? 'ru' : 'en'; // uk это украинский!!!
        const {errorIsShowed} = this.state;
        const formText = is2faEnabled
            ? i18n('Frontend.otp_input_description')
            : i18n('Frontend.auth.verify.password.subheader');
        const inputPlaceholderText = is2faEnabled ? i18n('Frontend.onetime_password') : i18n('Profile.domik.password');

        return (
            <div className='domik'>
                <div className={`domik__logo domik__logo__${domikLogoLang}`} />
                <form onSubmit={this.submitForm}>
                    <div className={classnames('domik__input', {'g-hide': !requestPassword})}>
                        <label className='domik__label domik__label_password' htmlFor='password'>
                            {formText}
                        </label>
                        <Input
                            id='password'
                            name='password'
                            type='password'
                            size='m'
                            controlRef={(password) => (this.password = password)}
                            value={this.state.password}
                            placeholder={inputPlaceholderText}
                            onChange={this.handleInput}
                        />
                        {errorIsShowed && errors.password && (
                            <div className='p-control-error-block'>{errorMessages.password[errors.password]}</div>
                        )}
                    </div>
                    {requestCaptcha && (
                        <div>
                            <div className='domik__captcha'>
                                <Captcha
                                    isMobile={isMobileCaptcha}
                                    lang={lang}
                                    introSound={captcha.introSound}
                                    captchaSound={captcha.captchaSound}
                                    imageUrl={captcha.imageUrl}
                                    loading={captcha.loading}
                                    loadingAudio={captcha.loadingAudio}
                                    playing={captcha.playing}
                                    type={captcha.type}
                                    dispatch={props.dispatch}
                                    visible={requestCaptcha}
                                />
                            </div>
                            <div className='domik__input'>
                                <label className='domik__label'>{i18n('Profile.domik.enter-symbols')}</label>
                                <Input
                                    id='captcha'
                                    name='captcha'
                                    size='m'
                                    controlRef={(captcha) => (this.captcha = captcha)}
                                    value={this.state.captcha}
                                    onChange={this.handleInput}
                                />
                                {errorIsShowed && errors.captcha && (
                                    <div className='p-control-error-block'>{errorMessages.captcha[errors.captcha]}</div>
                                )}
                            </div>
                        </div>
                    )}
                    <div className='button_type_submit-wrapper'>
                        <Button width='max' view='action' size='m' type='submit'>
                            {i18n('Profile.domik.confirm')}
                        </Button>
                    </div>
                    {requestPassword && (
                        <Button
                            view='pseudo'
                            size='m'
                            url='/passport?mode=restore'
                            type='link'
                            width={isMobile ? 'max' : 'auto'}
                        >
                            {i18n('Profile.domik.remember-password')}
                        </Button>
                    )}
                    <div
                        className={classnames('domik__cancel-button', {
                            'domik__wide-button': !requestPassword || isMobile,
                            'domik__narrow-button': requestPassword && !isMobile
                        })}
                    >
                        <Button onClick={this.hideDomik} view='pseudo' size='m' width={isMobile ? 'max' : 'auto'}>
                            {i18n('_AUTH_.common.cancel')}
                        </Button>
                    </div>
                </form>
            </div>
        );
    }
}

Domik.propTypes = {
    dispatch: PropTypes.func.isRequired,
    domik: PropTypes.shape({
        requestPassword: PropTypes.bool.isRequired,
        requestCaptcha: PropTypes.bool.isRequired,
        errors: PropTypes.object
    }).isRequired,
    captcha: PropTypes.object.isRequired,
    lang: PropTypes.string.isRequired,
    isMobileCaptcha: PropTypes.bool.isRequired,
    is2faEnabled: PropTypes.bool.isRequired,
    isMobile: PropTypes.bool.isRequired
};

class DomikModal extends Component {
    static mapStateToProps(state) {
        const {
            settings = {},
            access: {is2faEnabled}
        } = state;

        return {
            domik: state.domik,
            captcha: state.captcha,
            lang: settings.language,
            is2faEnabled,
            isMobileCaptcha: Boolean(settings.ua.isMobile || settings.ua.isTablet),
            isMobile: settings.ua && settings.ua.isMobile
        };
    }

    render() {
        const props = this.props;
        const {requestPassword, requestCaptcha} = props.domik;

        return (
            <Modal
                className='domik_password'
                visible={requestPassword || requestCaptcha}
                zIndexGroupLevel={90}
                dataT='domik'
            >
                {(requestPassword || requestCaptcha) && <Domik {...props} />}
            </Modal>
        );
    }
}

DomikModal.propTypes = {
    domik: PropTypes.shape({
        requestPassword: PropTypes.bool.isRequired,
        requestCaptcha: PropTypes.bool.isRequired
    }).isRequired,
    lang: PropTypes.string,
    isMobileCaptcha: PropTypes.bool,
    is2faEnabled: PropTypes.bool
};

const DomikModalConnected = connect(DomikModal.mapStateToProps)(DomikModal);

export {DomikModalConnected as Domik};
