import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Popup} from '@components/Popup';
import HeaderLogo from './HeaderLogo.jsx';
import ServicesNavigation from './services_navigation.jsx';
import Accounts from './accounts.jsx';
import metrics from '@blocks/metrics.js';
import {showUserDropDown} from './actions';

class Header extends Component {
    avatarRef = React.createRef();

    static mapStateToProps(state) {
        const {settings = {}, common = {}, header = {}, person = {}} = state;

        return {
            retpath: common.retpath,
            currentPage: common.currentPage,
            displayName: person.displayName,
            avatarId: header.defaultAccount.avatarId || '', // TODO move default value
            avatar: settings.avatar,
            isIntranet: settings.env.name === 'intranet',
            login: person.login,
            yandexuid: common.yandexuid,
            canAddMore: header.canAddMore,
            showUserDropdown: header.showUserDropdown,
            showSubNavigation: header.showSubNavigation,
            accounts: header.accounts,
            embeddedauth: settings.embeddedauth,
            tld: settings.tld,
            links: settings.links,
            isWSUser: common.isWSUser
        };
    }

    constructor(props) {
        super(props);

        this.openUserDropdown = this.openUserDropdown.bind(this);
        this.onOutsideClick = this.onOutsideClick.bind(this);
    }

    openUserDropdown(event) {
        event.preventDefault();
        this.props.dispatch(showUserDropDown(true));
    }

    onOutsideClick() {
        this.props.dispatch(showUserDropDown(false));
    }

    sendMetrics(msg) {
        metrics.send(`Шапка: ${msg}`);
    }

    getSplittedDisplayName() {
        const regexCodePoint = /[^\uD800-\uDFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDFFF]/g;
        const match = this.props.displayName && this.props.displayName.match(regexCodePoint);
        const result = {};

        if (match) {
            result.head = match[0];
            result.tail = match.slice(1).join('');
        }

        return result;
    }

    _maybeRenderServicesNavigation = () => {
        const {hideServicesNavigation, dispatch, isIntranet, isWSUser, tld, showSubNavigation, links} = this.props;

        if (hideServicesNavigation) {
            return null;
        }

        return (
            <ServicesNavigation
                dispatch={dispatch}
                isIntranet={isIntranet}
                isWSUser={isWSUser}
                tld={tld}
                showSubNavigation={showSubNavigation}
                links={links}
            />
        );
    };

    render() {
        const {
            avatar,
            login,
            currentPage,
            avatarId,
            yandexuid,
            retpath,
            canAddMore,
            accounts,
            embeddedauth,
            tld,
            showUserDropdown,
            links,
            isIntranet
        } = this.props;

        const splittedDisplayName = this.getSplittedDisplayName();
        const avatarPath = avatar.pathname
            .replace('%uid%', avatarId || '0/0-0')
            .replace('%login%', login)
            .replace('%size%', '50');

        const avatarStyle = {
            backgroundImage: `url("//${avatar.host}${avatarPath}")`
        };

        if (!accounts.length) {
            return (
                <div className='header'>
                    <HeaderLogo />
                    {this._maybeRenderServicesNavigation()}
                </div>
            );
        }

        return (
            <div className='header'>
                <a href={`/auth?retpath=${currentPage}`} className='header-user' onClick={this.openUserDropdown}>
                    <span className='header-user-displayname'>
                        <span className='header-user-displayname__firstletter'>{splittedDisplayName.head}</span>
                        {splittedDisplayName.tail}
                    </span>
                    <span className='header-user-avatar' style={avatarStyle} ref={this.avatarRef}>
                        <span className='header-user-avatar-mask' />
                    </span>
                </a>
                <Popup
                    target='anchor'
                    visible={showUserDropdown}
                    anchor={this.avatarRef}
                    directions={['bottom-right']}
                    onOutsideClick={this.onOutsideClick}
                    ignoreRefs={[this.avatarRef]}
                >
                    <Accounts
                        yu={yandexuid}
                        canAddMore={canAddMore}
                        accounts={accounts}
                        avatar={avatar}
                        embeddedauth={embeddedauth}
                        currentPage={currentPage}
                        retpath={retpath}
                        tld={tld}
                        isIntranet={isIntranet}
                        links={links}
                    />
                </Popup>
                <HeaderLogo />
                {this._maybeRenderServicesNavigation()}
            </div>
        );
    }
}

Header.propTypes = {
    dispatch: PropTypes.func.isRequired,
    language: PropTypes.string.isRequired,
    displayName: PropTypes.string.isRequired,
    avatarId: PropTypes.string.isRequired,
    isIntranet: PropTypes.bool.isRequired,
    avatar: PropTypes.shape({
        host: PropTypes.string.isRequired,
        pathname: PropTypes.string.isRequired
    }),
    login: PropTypes.string,
    yandexuid: PropTypes.string.isRequired,
    canAddMore: PropTypes.bool,
    accounts: PropTypes.arrayOf(PropTypes.object),
    embeddedauth: PropTypes.string.isRequired,
    retpath: PropTypes.string,
    currentPage: PropTypes.string.isRequired,
    hasComponentsForNav: PropTypes.bool,
    tld: PropTypes.string.isRequired,
    links: PropTypes.objectOf(PropTypes.string),
    isWSUser: PropTypes.bool,
    showUserDropdown: PropTypes.bool,
    showSubNavigation: PropTypes.bool,
    hideServicesNavigation: PropTypes.bool
};

const WrappedHeader = connect(Header.mapStateToProps)(Header);

export {WrappedHeader as Header};
