import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {goBack, replace} from 'connected-react-router';
import classnames from 'classnames';
import {connect} from 'react-redux';
import Link from '@blocks/morda/components/link/link.jsx';
import metrics from '@blocks/metrics.js';

const backButtonClassName = 'navigation-item navigation-back';

class Navigation extends Component {
    static mapStateToProps(state) {
        const {settings, common} = state;

        return {
            tld: settings.tld,
            isIntranet: settings.env.name === 'intranet',
            isWSUser: common.isWSUser,
            isPDD: common.isPDD,
            hasComponentsForNav: common.hasComponentsForNav,
            isMobileOrTouch: settings.ua.isMobile || settings.ua.isTouch || false,
            retpath: common.retpath,
            historyOnPassport: common.historyOnPassport,
            defaultPage: common.defaultPage,
            currentPage: common.currentPage
        };
    }

    constructor(props) {
        super(props);

        this.touchStart = this.touchStart.bind(this);
        this.touchMove = this.touchMove.bind(this);
        this.touchEnd = this.touchEnd.bind(this);
        this.goBackInHistory = this.goBackInHistory.bind(this);
    }

    componentWillMount() {
        const {isPDD, isWSUser, isIntranet, tld} = this.props;
        const isKUBR = ['kz', 'ua', 'by', 'ru'].indexOf(tld) !== -1;

        this.navigationItems = [
            {text: i18n('Navigation.nav.personalinfo'), url: '/profile', id: 'personal-info'},
            {text: i18n('Navigation.nav.access'), url: '/profile/access', id: 'access'},
            {text: i18n('Navigation.nav.emails'), url: '/profile/emails', id: 'emails'},
            {text: i18n('Navigation.nav.yasms'), url: '/profile/phones', id: 'phones'},
            {text: i18n('Navigation.nav.social'), url: '/profile/social', id: 'social'},
            {
                text: i18n('Navigation.nav.paymentinfo'),
                url: `//money.yandex.${tld}/card/card-payment/about`,
                id: 'paymentinfo'
            }
        ];

        if (!isKUBR || isPDD || isWSUser) {
            // ATTENTION!!! Яндекс.Деньги показываем только в КУБРе и не показываем ПДД и WS
            this.navigationItems.pop();
        }

        if (isWSUser) {
            // для пользователей WS не показываем соц. профили
            this.navigationItems.pop();
        }

        if (isIntranet) {
            this.navigationItems = [];
        }
    }

    componentDidMount() {
        this.nav = this.refs.navigation;
        this.maxScrollBounce = 50; // for scroll bounce effect
        this.calcElemsWidth();
        this.calcMaxMinusShift();

        $(window).on('resize', this.onResize.bind(this));
    }

    touchStart(event) {
        if (!event) {
            return;
        }

        if (this.disableScroll) {
            return;
        }

        if (event.nativeEvent.touches && event.nativeEvent.touches[0]) {
            this.startX = (event.nativeEvent.touches && event.nativeEvent.touches[0].pageX) || null;
        }
    }

    touchMove(event) {
        if (!event || this.disableScroll) {
            return;
        }

        if (event && typeof event.preventDefault === 'function') {
            event.preventDefault();
        }

        const nativeEvent = event.nativeEvent;
        const shift = ((nativeEvent.touches && nativeEvent.touches[0].pageX - this.startX) || 0) + this.currMargin;

        if (shift >= this.maxScrollBounce) {
            return;
        }

        if (shift <= this.maxMinusShift - this.maxScrollBounce) {
            return;
        }

        this.nav.style.marginLeft = `${shift}px`;
    }

    touchEnd() {
        if (this.disableScroll) {
            return;
        }

        this.startX = null;
        this.testNavPosition();
    }

    testNavPosition() {
        this.currMargin = parseInt(this.nav.style.marginLeft, 10) || 0;

        if (this.currMargin < this.maxMinusShift) {
            this.currMargin = this.maxMinusShift;
        }

        if (this.currMargin > 0 || this.disableScroll) {
            this.currMargin = 0;
        }

        this.nav.className = 'navigation__animate';
        this.nav.style.marginLeft = `${this.currMargin}px`;

        setTimeout(() => {
            this.nav.className = '';
        }, 300);
    }

    calcElemsWidth() {
        let elemsWidth = 0;

        $('.navigation-item', this.nav).each(function() {
            elemsWidth += $(this).outerWidth(true); // eslint-disable-line no-invalid-this
        });

        this.elemsWidth = elemsWidth;
    }

    calcMaxMinusShift() {
        const wrapperWidth = parseInt(this.refs['navigation-wrapper'].offsetWidth, 10) || 0;

        this.disableScroll = this.elemsWidth <= wrapperWidth;
        this.maxMinusShift = (this.elemsWidth - wrapperWidth + 10) * -1; // 10 - last margin
    }

    onResize() {
        setTimeout(() => {
            this.calcMaxMinusShift();
            this.testNavPosition();
        }, 10);
    }

    sendMetrics(msg) {
        metrics.send(`Верхняя навигация: ${msg}`);
    }

    goBackInHistory(event) {
        const {historyOnPassport, retpath, defaultPage} = this.props;

        // Если в истории 1 запись, то либо мы возвращаемся по retpath
        // либо идем на морду
        if (historyOnPassport > 0) {
            this.props.dispatch(goBack());
        } else if (retpath) {
            window.location.replace(retpath);
        } else {
            this.props.dispatch(replace(defaultPage)); // TODO проверить прокидывание параметров, использовать action
        }
        event.preventDefault();
    }

    getActiveNavigationItem() {
        let activeNavItem;
        const currentPage = this.props.currentPage;

        this.navigationItems.forEach((item) => {
            if (currentPage.indexOf(item.url) !== -1) {
                activeNavItem = item.id;
            }
        });

        return activeNavItem;
    }

    render() {
        const {retpath, defaultPage, isMobileOrTouch, hasComponentsForNav} = this.props;
        const backButtonUrl = retpath || defaultPage;
        const activeNavItem = this.getActiveNavigationItem();

        return (
            <div
                className='navigation'
                onTouchStart={this.touchStart}
                onTouchMove={this.touchMove}
                onTouchEnd={this.touchEnd}
            >
                {Boolean(isMobileOrTouch) && (
                    <a href={backButtonUrl} className={backButtonClassName} onClick={this.goBackInHistory} />
                )}

                <div
                    className={classnames('navigation-inner', {'navigation-inner__touch': isMobileOrTouch})}
                    ref='navigation-wrapper'
                >
                    <div ref='navigation'>
                        {this.navigationItems.map((item, index) => (
                            <span onMouseDown={() => this.sendMetrics(item.url)} key={index}>
                                <Link
                                    reactLink={hasComponentsForNav}
                                    key={item.id}
                                    className={classnames('navigation-item', {
                                        'navigation-item__active': item.id === activeNavItem
                                    })}
                                    url={item.url}
                                >
                                    {item.text}
                                </Link>
                            </span>
                        ))}
                    </div>
                </div>
            </div>
        );
    }
}

Navigation.propTypes = {
    tld: PropTypes.string.isRequired,
    hasNoComponentsForNav: PropTypes.bool,
    isIntranet: PropTypes.bool.isRequired,
    isWSUser: PropTypes.bool.isRequired,
    isPDD: PropTypes.bool.isRequired,
    isMobileOrTouch: PropTypes.bool.isRequired,
    retpath: PropTypes.string,
    historyOnPassport: PropTypes.number.isRequired,
    defaultPage: PropTypes.string.isRequired,
    currentPage: PropTypes.string.isRequired,
    dispatch: PropTypes.func.isRequired,
    hasComponentsForNav: PropTypes.bool
};

const WrappedNavigation = connect(Navigation.mapStateToProps)(Navigation);

export {WrappedNavigation as Navigation};
