import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {Modal} from '@components/Modal';
import {Link} from '@components/Link';
import {Button} from '@components/Button';
import {Popup} from '@components/Popup';
import {showHint} from '@blocks/morda/access/actions';
import {showRegPopup, setEditMode} from '@blocks/common/actions';

const TWOFA_ERROR_MODAL = 'twofaErrorModal';

export class TwoFA extends Component {
    hintAnchorRef = React.createRef();

    constructor(props) {
        super(props);
        this.displayName = 'twofa';
        this.showHint = this.showHint.bind(this);
        this.closeHint = this.closeHint.bind(this);
        this.closeErrorModal = this.closeErrorModal.bind(this);
        this.onLinkClick = this.onLinkClick.bind(this);
        this.setPopups = this.setPopups.bind(this);
    }

    showAppPasswordsModal() {} // eslint-disable-line no-empty-function

    componentDidMount() {
        this.setPopups();
    }

    setPopups() {
        const hintAnchor = this.refs['hint-anchor'] || null;
        const hintPopup = this.refs['hint-popup'] || null;

        if (hintAnchor && hintPopup) {
            hintPopup.setAnchor(hintAnchor);
        }
    }

    closeErrorModal() {
        this.props.dispatch(setEditMode(null));
    }

    sendMetrics(msg) {
        this.props.sendMetrics(`Включить вход по одноразовым паролям: ${msg}`);
    }

    showHint() {
        if (this.props.isTouch || this.props.isEnabled) {
            return;
        }
        this.setPopups();
        this.props.dispatch(showHint(this.displayName));
        this.sendMetrics('показ попапа');
    }

    closeHint() {
        if (this.props.isTouch || this.props.isEnabled) {
            return;
        }

        this.props.dispatch(showHint(''));
    }

    onLinkClick() {
        const props = this.props;
        const {passwordStrength, dispatch, canChangePassword, is2faEnabled} = props;
        const url = props.is2faEnabled ? props.disableLink : props.enableLink;

        if (canChangePassword !== undefined && !canChangePassword) {
            this.props.dispatch(setEditMode(TWOFA_ERROR_MODAL));
        } else if (passwordStrength === -1 && !is2faEnabled) {
            dispatch(showRegPopup(true));
        } else {
            window.location.href = url;
        }
    }

    render() {
        const {isEnabled, settings, enabledStateMenu} = this.props;
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;

        return (
            <div className='section-wrapper'>
                {isEnabled ? (
                    <div>
                        <div className='p-control-link p-control-link__bold' ref={this.hintAnchorRef}>
                            <span className='link' onMouseDown={() => this.sendMetrics('переход по ссылке')}>
                                <span>{enabledStateMenu.title}</span>
                            </span>
                        </div>
                        <div className='p-control-link' data-t='access-widget:twofa-change-device'>
                            <Link href={this.props.actionLink}>{enabledStateMenu.actionText}</Link>
                        </div>
                        <div className='p-control-link' data-t='access-widget:twofa-disable'>
                            <Link href={this.props.disableLink}>{enabledStateMenu.disableText}</Link>
                        </div>
                    </div>
                ) : (
                    <div>
                        <div
                            className='p-control-link p-control-link__bold'
                            data-t='access-widget:twofa-enable'
                            ref={this.hintAnchorRef}
                        >
                            <Link pseudo={true} onClick={this.onLinkClick}>
                                {isEnabled ? this.props.disableText : this.props.enableText}
                            </Link>
                            {!isPhone && (
                                <span>
                                    <span
                                        className='show-hint__icon'
                                        onClick={this.showHint}
                                        onMouseDown={() => this.sendMetrics('переход по ссылке')}
                                    />
                                    {this.getHintPopup()}
                                </span>
                            )}
                        </div>
                        <div className='section-body-info section-body-info__bottom_shift'>
                            {this.props.descriptionMessage}
                        </div>
                    </div>
                )}
                <Modal visible={this.props.setEditMode === TWOFA_ERROR_MODAL} zIndexGroupLevel={-0.5} dataT='twofa'>
                    <div className='p-control-modal'>
                        <div className='p-control-modal-close' onClick={this.closeErrorModal} />
                        <div className='section-head'>
                            <h2 className='section-title'>{i18n('Errors.2fa.unavailable')}</h2>
                        </div>
                        <div className='p-control-modal-message'>{i18n('Mend.2fa.unavailable.message')}</div>
                    </div>
                </Modal>
            </div>
        );
    }

    getHintPopup() {
        return (
            <Popup
                target='anchor'
                visible={this.props.featureHint === this.displayName}
                anchor={this.hintAnchorRef}
                directions={['top-center', 'top-right']}
                onOutsideClick={this.closeHint}
                ignoreRefs={[this.hintAnchorRef]}
            >
                <div className={`p-control-modal p-control-modal__hint p-control-modal__hint__${this.displayName}`}>
                    <div className='p-control-modal-close' onClick={this.closeHint} />
                    {this.getHintPopupContent()}
                    <span onMouseDown={() => this.sendMetrics('настроить в попапе')}>
                        <Button
                            size='l'
                            view='action'
                            type='button'
                            onClick={this.onLinkClick}
                            text={i18n('Frontend.tune')}
                        />
                    </span>
                </div>
            </Popup>
        );
    }

    getHintPopupContent() {
        return (
            <div>
                <div className='section-title'>{this.props.hintTitle}</div>
                <div className='hint-message'>{this.props.hintMessage}</div>
            </div>
        );
    }
}

TwoFA.defaultProps = {
    enableText: i18n('Profile.twofa.enable'),
    disableText: i18n('Profile.twofa.disable'),
    enableLink: '/profile/access/2fa?origin=passport_profile',
    disableLink: '/profile/access/2fa/disable?origin=passport_profile',
    actionLink: '/profile/access/2fa/migrate',
    hintTitle: i18n('Profile.twofa.hint.title'),
    hintMessage: i18n('Profile.twofa.hint.message'),
    descriptionMessage: i18n('Profile.twofa.description.message'),
    enabledStateMenu: {
        title: i18n('Profile.access.2fa.title'),
        actionText: i18n('Profile.access.2fa.change-device'),
        disableText: i18n('Profile.common.disable')
    }
};

TwoFA.propTypes = {
    setEditMode: PropTypes.string,
    appPasswordsCount: PropTypes.number,
    actionLink: PropTypes.string,
    disableLink: PropTypes.string,
    sendMetrics: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    isEnabled: PropTypes.bool.isRequired,
    isTouch: PropTypes.bool,
    featureHint: PropTypes.string,
    disableText: PropTypes.string,
    enableText: PropTypes.string,
    hintTitle: PropTypes.string,
    hintMessage: PropTypes.string,
    descriptionMessage: PropTypes.string,
    enabledStateMenu: PropTypes.shape({
        title: PropTypes.string,
        actionText: PropTypes.string,
        disableText: PropTypes.string
    }),
    settings: PropTypes.shape({
        ua: PropTypes.shape({
            isMobile: PropTypes.bool,
            isTouch: PropTypes.bool,
            isTablet: PropTypes.bool
        }).isRequired
    }),
    passwordStrength: PropTypes.number,
    canChangePassword: PropTypes.bool,
    is2faEnabled: PropTypes.bool,
    enableLink: PropTypes.string
};
