import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {push} from 'react-router-redux';
import classnames from 'classnames';
import {Link} from '@components/Link';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';
import {Input} from '@components/Input';
import {setEditMode} from '@blocks/common/actions';
import {updateAppPwdData} from './actions';
import {getClientInfo, createAppPassword} from './';

const clientNames = {
    mail: {
        name: i18n('Profile.access.apppwd-mail-dative'),
        fieldPlaceholder: i18n('Profile.access.apppwd-mail-placeholder')
    },
    disk: {
        name: i18n('Profile.access.appwd-files-dative'),
        fieldPlaceholder: i18n('Profile.access.apppwd-disk-placeholder')
    },
    addrbook: {
        name: i18n('Profile.access.appwd-contacts-dative'),
        fieldPlaceholder: i18n('Profile.access.apppwd-addrbook-placeholder')
    },
    calendar: {
        name: i18n('Profile.access.appwd-calendar-dative'),
        fieldPlaceholder: i18n('Profile.access.apppwd-calendar-placeholder')
    },
    magnitola: {
        name: i18n('Profile.access.appwd-magnitola-dative'),
        fieldPlaceholder: i18n('Profile.access.apppwd-magnitola-placeholder')
    },
    chat: {
        name: i18n('Profile.access.appwd-chat-dative'),
        fieldPlaceholder: i18n('Profile.access.apppwd-chat-placeholder')
    }
};

export class CreateAppPasswordBlock extends Component {
    constructor(props) {
        super(props);

        this.closeModal = this.closeModal.bind(this);
        this.passDeviceName = this.passDeviceName.bind(this);
        this.generatePassword = this.generatePassword.bind(this);
        this.clearType = this.clearType.bind(this);
        this.changeDeviceName = this.changeDeviceName.bind(this);

        this.state = {
            deviceName: ''
        };
    }

    componentDidMount() {
        this.props.dispatch(getClientInfo());
    }

    closeModal() {
        const defaults = {
            clientId: '',
            deviceName: '',
            clientName: '',
            passwordCreated: ''
        };
        const {settings, dispatch} = this.props;
        const isStandAlonePage = this.props.common && this.props.common.currentPage !== '/profile';
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;

        dispatch(updateAppPwdData(defaults));
        if (isPhone || isStandAlonePage) {
            dispatch(push('/profile'));
        } else {
            dispatch(setEditMode(null));
        }
    }

    passDeviceName(event) {
        let passNameValue = event.target.value && event.target.value.trim();

        if (passNameValue.length > 100) {
            passNameValue = event.target.value.trim().slice(0, 100);
        }

        this.props.dispatch(updateAppPwdData({deviceName: passNameValue}));
    }

    generatePassword() {
        this.props.dispatch(createAppPassword());
    }

    clearType() {
        const defaults = {
            clientId: '',
            clientName: ''
        };

        this.props.dispatch(updateAppPwdData(defaults));
    }

    changeDeviceName(event) {
        const {target = {}} = event;
        const {value} = target;

        this.setState({deviceName: value});
    }

    render() {
        const {settings, dispatch, appPasswords, modal} = this.props;
        const createAppPassword = appPasswords.createAppPassword;
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;
        const isAppPasswordsEnabled = modal
            ? this.props.isAppPasswordsEnabled
            : this.props.access && this.props.access.isAppPasswordsEnabled;
        const showInput = createAppPassword && createAppPassword.clientId;
        const clientLocName =
            createAppPassword &&
            clientNames[createAppPassword.clientName] &&
            clientNames[createAppPassword.clientName].name;
        const createdPassword = createAppPassword && createAppPassword.passwordCreated;
        const appPwdAboutLink = `https://yandex.${settings.tld}/support/passport/authorization/app-passwords.xml`;
        const fieldPlaceholder =
            createAppPassword &&
            clientNames[createAppPassword.clientName] &&
            clientNames[createAppPassword.clientName].fieldPlaceholder;
        const errorMessage = createAppPassword.error;

        return (
            <div className={classnames('p-control-form', {'p-control-modal': modal})}>
                {Boolean(!isPhone) && (
                    <div>
                        <h2 className='section-title'>{i18n('Profile.access.create-apppwd')}</h2>
                        <div className='section-info'>{i18n('Profile.access.apppwd-description')}</div>
                    </div>
                )}

                {createdPassword && createdPassword !== 'processing' ? (
                    <div>
                        <PasswordCreated
                            isPhone={isPhone}
                            os={settings.ua.OSFamily}
                            clientLocName={clientLocName}
                            createAppPassword={createAppPassword}
                            dispatch={dispatch}
                        />
                        <div className='create_app_pwd-button-container'>
                            <div className='create_app_pwd-button-link'>
                                <Link target='_blank' href={appPwdAboutLink}>
                                    {i18n('Profile.access.apppwd-about-link')}
                                </Link>
                            </div>
                            <div className='create_app_pwd-button-done'>
                                {isPhone ? (
                                    <Button
                                        text={i18n('Profile.common.done')}
                                        size='l'
                                        type='link'
                                        view='action'
                                        url='/profile'
                                    />
                                ) : (
                                    <Button
                                        text={i18n('Profile.common.done')}
                                        size='l'
                                        view='action'
                                        onClick={this.closeModal}
                                    />
                                )}
                            </div>
                        </div>
                    </div>
                ) : (
                    <div className='create_app_pwd-content'>
                        {(() => {
                            if (isPhone && !showInput) {
                                return (
                                    <div>
                                        <h2 className='section-title'>{i18n('Profile.access.apppwd-app-type')}</h2>
                                        <p className='gray'>{i18n('Profile.access.apppwd.select-type')}</p>
                                    </div>
                                );
                            }

                            if (isPhone && showInput) {
                                return (
                                    <h2 className='section-title'>
                                        {i18n('Profile.access.appwd-getpass-title').replace('%s', clientLocName)}
                                    </h2>
                                );
                            }

                            return (
                                <div>
                                    <h3>{i18n('Profile.access.apppwd.select-type-title')}</h3>
                                    <p className='gray'>{i18n('Profile.access.apppwd.select-type')}</p>
                                </div>
                            );
                        })()}

                        {Boolean((isPhone && !showInput) || !isPhone) && (
                            <AppTypes createAppPassword={createAppPassword} dispatch={dispatch} />
                        )}

                        {Boolean(showInput) && (
                            <div className='create_app_pwd-device-name-block'>
                                <h3>{i18n('Profile.access.appwd-setname-title').replace('%s', clientLocName)}</h3>
                                <div className='create_app_pwd-device-name-form'>
                                    <Input
                                        size='l'
                                        type='text'
                                        value={this.state.deviceName}
                                        onChange={this.changeDeviceName}
                                        placeholder={fieldPlaceholder}
                                        name='app_pwd-device-name'
                                        onBlur={this.passDeviceName}
                                    />
                                </div>
                                <div className='input-error'>{errorMessage}</div>
                                {Boolean(!isAppPasswordsEnabled) && (
                                    <div className='create_app_pwd-warning-block'>
                                        {Boolean(!isPhone) && (
                                            <h4 className='warning'>{i18n('Profile.access.appwd.warning-title')}</h4>
                                        )}
                                        <p>{i18n('Profile.access.apppwd.warning-text')}</p>
                                    </div>
                                )}
                            </div>
                        )}

                        {Boolean((isPhone && showInput) || !isPhone) && (
                            <div>
                                <div className='create_app_pwd-button-container'>
                                    {Boolean(showInput) && (
                                        <div className='create_app_pwd-button-generate'>
                                            <div className='personal-spinner'>
                                                <Spin size='s' progress={createdPassword === 'processing'} />
                                            </div>
                                            <Button
                                                text={i18n('Profile.access.apppwd-generate')}
                                                size='l'
                                                view='action'
                                                onClick={this.generatePassword}
                                            />
                                        </div>
                                    )}

                                    <div className='create_app_pwd-button-cancel'>
                                        <Button
                                            text={i18n('_AUTH_.common.cancel')}
                                            size='l'
                                            view='pseudo'
                                            onClick={this.closeModal}
                                        />
                                    </div>
                                </div>

                                {Boolean(isPhone) && (
                                    <Link pseudo={true} onClick={this.clearType}>
                                        {i18n('Profile.apppasswords.change-type')}
                                    </Link>
                                )}
                            </div>
                        )}
                    </div>
                )}
            </div>
        );
    }
}

class AppTypes extends Component {
    constructor(props) {
        super(props);
        this.handleClick = this.handleClick.bind(this);
    }

    handleClick(e) {
        const element = e.currentTarget;
        const clientInfo = {
            clientId: element.getAttribute('data-client'),
            clientName: element.getAttribute('data-name')
        };

        this.props.dispatch(updateAppPwdData(clientInfo));
    }

    render() {
        const {createAppPassword} = this.props;
        // PASSP-21582
        const clientsList = createAppPassword && createAppPassword.clients.filter((client) => client.name !== 'chat');
        const clientId = createAppPassword && createAppPassword.clientId;

        return (
            <div>
                {Boolean(clientsList && clientsList.length > 0) && (
                    <ul className='create_app_pwd-items'>
                        {clientsList.map((client) => (
                            <li
                                className={
                                    clientId === client.id
                                        ? `item active type-${client.name}`
                                        : `item type-${client.name}`
                                }
                                key={client.id}
                                data-client={client.id}
                                data-name={client.name}
                                onClick={this.handleClick}
                            >
                                <span className={`app_pwd-icon ${client.name}-icon`}>
                                    <span className='icon' />
                                    {/* Hello, Opera 12 svg-bg and border-radius! */}
                                </span>
                                <strong className='name'>{client.locName}</strong>
                                <span className='protocol gray'>{client.protocol}</span>
                            </li>
                        ))}
                    </ul>
                )}
            </div>
        );
    }
}

class PasswordCreated extends Component {
    constructor(props) {
        super(props);
        this.copyResult = this.copyResult.bind(this);
    }

    copyResult() {
        const copiedMessageElem = document.querySelector('.create_app_pwd-copy-message');

        this.refs.resultText.select();
        document.execCommand('copy');
        copiedMessageElem.classList.add('visible');
        setTimeout(() => {
            copiedMessageElem.classList.remove('visible');
        }, 2000);
    }

    render() {
        const {createAppPassword, isPhone, os} = this.props;
        const passwordText = createAppPassword && createAppPassword.passwordCreated;
        const passwordName =
            createAppPassword && createAppPassword.deviceName ? `«${createAppPassword.deviceName}»` : '';

        let showCopyBtn = false;

        try {
            document.execCommand('copy');
            showCopyBtn = true;
        } catch (err) {
            /* eslint no-empty: "error" */
            // continue regardless of error
        }
        if (os === 'iOS') {
            showCopyBtn = false;
        }
        return (
            <div className='create_app_pwd-content'>
                {Boolean(isPhone) && (
                    <h2 className='section-title'>
                        {i18n('Profile.access.appwd-getpass-title').replace('%s', this.props.clientLocName)}
                    </h2>
                )}
                <h3 className='create_app_pwd-pass-wrapper'>
                    {i18n('Profile.access.apppwd-copy-pass').replace('%s', passwordName)}
                </h3>
                <div className='create_app_pwd-created-password'>
                    <span className='create_app_pwd-created-text'>{passwordText}</span>
                    {Boolean(showCopyBtn) && (
                        <div className='create_app_pwd-button-copy'>
                            <Button
                                view='pseudo'
                                text={i18n('Profile.access.apppwd-copy-btn')}
                                size='l'
                                onClick={this.copyResult}
                            />
                            <input
                                className='helper-input'
                                readOnly='readonly'
                                name='result-text'
                                value={passwordText}
                                ref='resultText'
                            />
                        </div>
                    )}
                </div>

                {Boolean(showCopyBtn) && (
                    <span className='create_app_pwd-copy-message'>{i18n('Profile.access.appwd.pass-copied')}</span>
                )}

                {Boolean(!this.props.isPhone) && <h3>{i18n('Profile.access.apppwd-howto-tile')}</h3>}

                <div className='create_app_pwd-text-block'>{i18n('Profile.access.apppwd-howto-text')}</div>
            </div>
        );
    }
}

CreateAppPasswordBlock.propTypes = {
    appPasswords: PropTypes.shape({
        createAppPassword: PropTypes.object,
        tokens: PropTypes.object
    }),
    access: PropTypes.object,
    dispatch: PropTypes.func,
    edit: PropTypes.string,
    isAppPasswordsEnabled: PropTypes.bool,
    modal: PropTypes.bool,
    settings: PropTypes.object,
    common: PropTypes.shape({
        currentPage: PropTypes.string
    }).isRequired
};

AppTypes.propTypes = {
    createAppPassword: PropTypes.shape({
        clients: PropTypes.array.isRequired,
        clientId: PropTypes.string,
        deviceName: PropTypes.string,
        clientName: PropTypes.string,
        passwordCreated: PropTypes.string
    }).isRequired,
    dispatch: PropTypes.func.isRequired
};

PasswordCreated.propTypes = {
    createAppPassword: PropTypes.shape({
        clients: PropTypes.array.isRequired,
        clientId: PropTypes.string,
        deviceName: PropTypes.string,
        clientName: PropTypes.string,
        passwordCreated: PropTypes.string
    }),
    clientLocName: PropTypes.string,
    os: PropTypes.string,
    isPhone: PropTypes.bool,
    dispatch: PropTypes.func.isRequired
};
