import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import {MY} from '../../../config';
import metrics from '@blocks/metrics';

const translations = [i18n('Favorites.afisha.cinema.one'), i18n('Favorites.afisha.cinema.many')];

const onDownload = () => metrics.send([MY, 'Нажатие на Скачать билет']);
const onCancel = () => metrics.send([MY, 'Нажатие на Вернуть билет']);

export default function AfishaCard(isPhone) {
    const _AfishaCard = ({item}) => {
        const typeClass = isPhone ? '-touch' : '-desktop';
        const totalTicketCount = item.total_ticket_count;

        return (
            <div
                className={classnames('d-card-item afisha-card__item', {
                    'afisha-card_passed': item.sessionPassed
                })}
            >
                {item.image && (
                    <div
                        className={`afisha-card__image d-card-image afisha-card__image${typeClass}`}
                        style={{
                            backgroundImage: `url("${item.image}")`
                        }}
                    />
                )}
                <div
                    className={classnames(`d-card-content afisha-card__content${typeClass}`, {
                        'd-card-content_no-image': !item.image
                    })}
                >
                    {item.typeName && <div className='afisha-card__type'>{item.typeName}</div>}
                    <div className='d-card-title afisha-card__title'>
                        {item.links.event ? (
                            <a href={item.links.event} data-t='afisha-card:event-link' className='d-link d-link_black'>
                                {item.eventName}
                            </a>
                        ) : (
                            item.eventName
                        )}
                    </div>
                    <div className='afisha-card__date'>
                        {item.date}
                        {item.endDate && (
                            <span>
                                <span className='capitalize'>{i18n('Dashboard.to')}</span> {item.endDate}
                            </span>
                        )}
                    </div>
                    <div className='afisha-card__area'>
                        {item.links.venue && (
                            <a href={item.links.venue} data-t='afisha-card:area-link' className='d-link'>
                                {item.hallName}
                            </a>
                        )}
                        {!item.links.venue &&
                            (item.places > 0
                                ? translations[item.places % 100 === 1 ? 0 : 1].replace('%1', item.places)
                                : item.hallName)}
                    </div>
                    {totalTicketCount && (
                        <div className='afisha-card__tickets'>
                            <span data-t='afisha-card:hallname' className='font-weight_bold'>
                                {item.venueName ? `${item.venueName}: ` : ''}
                            </span>
                            <span data-t='afisha-card:ticket-count'>
                                {item.total_ticket_count}{' '}
                                {i18n({
                                    count: totalTicketCount,
                                    one: 'Dashboard.afisha.tickets.one',
                                    some: 'Dashboard.afisha.tickets.some',
                                    many: 'Dashboard.afisha.tickets.many',
                                    none: 'Dashboard.afisha.tickets.many'
                                })}
                            </span>
                        </div>
                    )}
                    {item.links.tickets && (
                        <div className='afisha-card__links'>
                            <a
                                href={item.links.tickets}
                                data-t='afisha-card:tickets-link'
                                className='d-link afisha-card__download afisha-card__download_margin-right'
                                onMouseDown={onDownload}
                            >
                                {i18n('Dashboard.afisha.download')}
                            </a>
                            <a
                                href='https://widget.afisha.yandex.ru/refund'
                                data-t='afisha-card:tickets-cancel'
                                className='d-link afisha-card__cancel'
                                onMouseDown={onCancel}
                            >
                                {i18n('Dashboard.afisha.cancel')}
                            </a>
                        </div>
                    )}
                    {item.sessionPassed && (
                        <span className='afisha-card__info'>{i18n('Dashboard.afisha.completed')}</span>
                    )}
                </div>
            </div>
        );
    };

    _AfishaCard.propTypes = {
        item: PropTypes.shape({
            date: PropTypes.string,
            endDate: PropTypes.string,
            typeName: PropTypes.string,
            hallName: PropTypes.string,
            venueName: PropTypes.string,
            image: PropTypes.string.isRequired,
            eventName: PropTypes.string.isRequired,
            places: PropTypes.number,
            total_ticket_count: PropTypes.number,
            sessionPassed: PropTypes.bool,
            links: PropTypes.shape({
                event: PropTypes.string,
                venue: PropTypes.string,
                tickets: PropTypes.string
            }).isRequired
        }).isRequired
    };

    return _AfishaCard;
}
