import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import debounce from 'lodash/debounce';

import Arrow from '../arrow/arrow.jsx';
import ArrowLink from '../arrow_link/arrow_link.jsx';

import listen from '@blocks/common/event_listeners';

import * as extracted from './scroll.js';

export default class Scroll extends React.Component {
    constructor(props) {
        super(props);

        this.state = {
            scroll: 0,
            mounted: false
        };
        this.cardCount = this.props.cardCount;
        this.scroll = extracted.onScroll.bind(this);

        extracted.construct.call(this, props);
    }

    componentDidMount() {
        this.destroyListener = listen('resize', debounce(extracted.updateScroll.bind(this), 600));
        setTimeout(() => {
            this.setState({
                mounted: true
            });
        }, 0);
    }

    componentWillUnmount() {
        if (this.destroyListener) {
            this.destroyListener();
        }
    }

    shouldComponentUpdate(_, nextState) {
        return this.state !== nextState;
    }

    render() {
        const max = extracted.getMax.call(this);
        const {isPhone, items, Card, link} = this.props;
        const scroll = !this.props.isPhone ? Math.min(max, this.state.scroll) : 0;
        const onRight = scroll >= max;

        return (
            <div
                className={classnames('d-cards', {
                    'd-cards-touch': isPhone
                })}
            >
                {!isPhone && link && (
                    <a
                        href={link.url}
                        className={classnames('d-cards-link d-link', {
                            'd-cards-link_hidden': onRight
                        })}
                        onMouseDown={this.sendMetrics}
                    >
                        {link.text}
                    </a>
                )}
                <div className={`d-cards__wrapper d-cards__wrapper-${isPhone ? 'touch' : 'desktop'}`}>
                    <div
                        className={`d-cards__inner d-cards__inner-${isPhone ? 'touch' : 'desktop'}`}
                        style={{transform: `translateX(-${(scroll * 102) / this.cardCount}%)`}}
                    >
                        <div className={`d-cards__content d-cards__content-${isPhone ? 'touch' : 'desktop'}`}>
                            {items.map((item, ndx) => (
                                <div
                                    className={classnames('d-card dashboard__anim', {
                                        [`d-card__${this.props.cardCount === 2 ? 'wide' : 'normal'}`]: !isPhone,
                                        'd-card-touch': isPhone,
                                        dashboard__anim_faded: this.state.mounted
                                    })}
                                    key={ndx}
                                >
                                    <Card item={item} />
                                </div>
                            ))}
                            {link && (
                                <ArrowLink
                                    href={link.url}
                                    text={link.text}
                                    onMouseDown={this.sendMetrics}
                                    className={classnames('dashboard__anim', {
                                        dashboard__anim_faded: this.state.mounted
                                    })}
                                />
                            )}
                        </div>
                    </div>
                </div>
                {!isPhone && (
                    <div>
                        <Arrow hidden={scroll === 0} direction='left' onClick={this.scroll} />
                        <Arrow hidden={onRight} onClick={this.scroll} />
                    </div>
                )}
            </div>
        );
    }
}

Scroll.propTypes = {
    Card: PropTypes.func.isRequired,
    items: PropTypes.array.isRequired,
    isPhone: PropTypes.bool.isRequired,
    metrics: PropTypes.arrayOf(PropTypes.string),
    cardCount: PropTypes.number,
    link: PropTypes.shape({
        url: PropTypes.string.isRequired,
        text: PropTypes.string.isRequired
    })
};

Scroll.defaultProps = {
    cardCount: 2
};
