import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import listen from '@blocks/common/event_listeners';

import * as extracted from './scroll_to_service.js';

const serviceNames = {
    social: i18n('Profile2.titles.social'),
    plus: i18n('Dashboard.plus.title.short'),
    key: i18n('Profile2.titles.passes_auth'),
    card: i18n('Profile2.titles.bank_cards'),
    cloud: i18n('common.serviceTitles.cloud'),
    music: i18n('common.serviceTitles.music'),
    money: i18n('common.serviceTitles.money'),
    video: i18n('common.serviceTitles.video'),
    mail: i18n('Profile2.titles.mails_phones'),
    afisha: i18n('common.serviceTitles.afisha'),
    market: i18n('common.serviceTitles.market'),
    maps: i18n('Profile.addresses.title.short'),
    'maps-bookmarks': i18n('Dashboard.maps.title'),
    clock: i18n('Profile2.titles.history_devices'),
    collections: i18n('common.serviceTitles.collections'),
    kp: i18n('Subscriptions.showcase.group.kp'),
    disk: i18n('Subscriptions.showcase.group.disk'),
    beru: i18n('Subscriptions.showcase.group.beru'),
    station: i18n('Subscriptions.showcase.group.station'),
    'other-settings': i18n('Profile2.titles.other_settings.short'),
    'new-plus': i18n('Dashboard.plus.title.short')
};

export default class ScrollToService extends React.Component {
    constructor(props) {
        super(props);

        this.state = {
            active: props.services[0],
            fixed: false
        };
        this.handlers = {};
        this.handleScroll = extracted.handleScroll.bind(this);
        extracted.constructHandlers.call(this, props.services);
        this.disabled = false;
    }

    componentDidMount() {
        if (typeof window !== 'undefined' && this.props.isMobile) {
            const container = $('<div/>')
                .css({height: 20, overflow: 'scroll', position: 'absolute', left: -999})
                .append($('<div/>').css('height', 100))
                .appendTo('body')
                .scrollTop(10);

            this.disabled = container.scrollTop() === 0;
            container.remove();
        }

        if (!this.disabled) {
            if (this.props.activated) {
                window.scrollTo(0, 0); // to prevent native anchor's scrolling

                if (this.props.plusLoaded) {
                    extracted.scrollToActivatedService.call(this);
                }
            } else {
                this.handleScroll();
            }
            this.destroyListener = listen('scroll', this.handleScroll);
        }

        this.forceUpdate();
    }

    componentWillUnmount() {
        if (this.destroyListener) {
            this.destroyListener();
        }
    }

    shouldComponentUpdate({services, plusLoaded}, nextState) {
        if (this.disabled) {
            return false;
        }

        if (plusLoaded && !this.props.plusLoaded) {
            return true;
        }

        return this.props.services.length !== services.length || this.state !== nextState;
    }

    componentDidUpdate(prevProps) {
        if (this.props.activated && !prevProps.plusLoaded && this.props.plusLoaded) {
            extracted.scrollToActivatedService.call(this);
        }
    }

    componentWillReceiveProps({services}) {
        extracted.constructHandlers.call(this, services);
    }

    render() {
        if (this.disabled) {
            return null;
        }

        return (
            <div className='sts__wrapper'>
                <div className={'sts'}>
                    <ul className='sts__inner'>
                        {this.props.services.map((service) => (
                            <li
                                className={classnames('sts__service', {
                                    sts__service_active: this.state.active === service
                                })}
                                onClick={this.handlers[service]}
                                key={service}
                                role='link'
                                tabIndex='0'
                            >
                                <span className={`sts__service-icon d-icon-${service} p-icon__${service}`} />
                                <span className='sts__service-name'>{serviceNames[service]}</span>
                            </li>
                        ))}
                    </ul>
                </div>
            </div>
        );
    }
}

ScrollToService.defaultProps = {
    prefix: '',
    sectionClassName: 'dashboard__section'
};

ScrollToService.propTypes = {
    prefix: PropTypes.string.isRequired,
    sectionClassName: PropTypes.string.isRequired,
    isMobile: PropTypes.bool,
    forceListen: PropTypes.bool,
    services: PropTypes.array.isRequired,
    activated: PropTypes.string,
    plusLoaded: PropTypes.bool
};
