import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import metrics from '@blocks/metrics';
import {hasExp} from '@blocks/selectors';

import Plus from './blocks/plus/plus.jsx';
import Music from './blocks/music/music.jsx';
import Cloud from './blocks/cloud/cloud.jsx';
import Afisha from './blocks/afisha/afisha.jsx';
import Market from './blocks/market/market.jsx';
import Collections from './blocks/collections/collections.jsx';
import MapsBookmarks from './blocks/maps_bookmarks/maps_bookmarks.jsx';

import ScrollToService from './components/scroll_to_service/scroll_to_service.jsx';

import navigationFactory, {MY} from './config';

import * as extracted from './dashboard.js';

/* {
    space: `+10 ${i18n('Dashboard.disk.space.gb')}`,
    price: <span>30 <span className='rub'>Р</span> {i18n('Dashboard.disk.cost.monthly')}</span>,
    url: navigation.disk.pay[0],
    onMouseDown: () => metrics.send([MY, 'Нажатие на + 10 гб'])
}, */

const restServices = ['afisha', 'market', 'collections', 'maps-bookmarks'];

export default class Dashboard extends React.Component {
    constructor(props) {
        super(props);

        const navigation = (this.navigation = navigationFactory((props.settings || {}).tld || 'ru'));

        this.cloudItems = [
            {
                space: `+100 ${i18n('Dashboard.disk.space.gb')}`,
                price: (
                    <span>
                        80 <span className='rub'>Р</span> {i18n('Dashboard.disk.cost.monthly')}
                    </span>
                ),
                url: navigation.disk.pay[1],
                onMouseDown: () => metrics.send([MY, 'Нажатие на + 100 гб'])
            },
            {
                space: `+1 ${i18n('Dashboard.disk.space.tb')}`,
                price: (
                    <span>
                        200 <span className='rub'>Р</span> {i18n('Dashboard.disk.cost.monthly')}
                    </span>
                ),
                url: navigation.disk.pay[2],
                onMouseDown: () => metrics.send([MY, 'Нажатие на + 1 тб'])
            }
        ];
        this.isRU = props.settings.tld === 'ru';
    }

    componentDidMount() {
        extracted.getDataHandler.call(this);
    }

    /* eslint-disable complexity */
    render() {
        const {
            person,
            dashboard,
            publicId,
            common: {locationCountryId},
            settings: {ua = {}}
        } = this.props;
        const plus = dashboard.plus;
        const {
            diskData,
            musicData,
            afishaData,
            favAfishaData,
            favMarketData,
            collectionsData,
            mapsBookmarksData
        } = dashboard;
        const isMusicVisible = extracted.isMusicVisible(this.props);
        const plusLoaded = !plus.enabled;
        const isPhone = extracted.isPhone.call(this);
        const isPhoneOrTablet = extracted.isPhone.call(this, false);
        const navigation = this.navigation;
        const invalidCloud = diskData.errors !== undefined && diskData.errors[0] === 'account.invalid_type';
        const services = [];
        const innerCN = classnames('dashboard__inner', {
            dashboard__inner_touch: isPhoneOrTablet
        });
        const hasPlusUpgradeExp = hasExp(this.props, 'plus-redesign-enable');

        const isYandexIosApp =
            (ua.BrowserName === 'YandexSearch' || ua.BrowserName === 'YandexBrowser') && ua.OSFamily === 'iOS';
        const showSubscriptionsCountryIds = [
            10099, // Монголия
            168, // Армения
            167, // Азербайджан
            149, // Беларусь
            159, // Казахстан
            207, // Кыргызстан
            208, // Молдова
            209, // Таджикистан
            171, // Узбекистан
            181, // Израиль
            225 // Россия
        ];

        const hideSubscriptions = isYandexIosApp && !showSubscriptionsCountryIds.includes(locationCountryId);

        if (plus.allowed || plus.enabled) {
            services.push(hasPlusUpgradeExp ? 'new-plus' : 'plus');
        }

        if (!person.isSocialchik && !invalidCloud) {
            services.push('cloud');
        }

        if (isMusicVisible) {
            services.push('music');
        }

        const allServices = services.concat(restServices);
        const activatedService = extracted.pickActivatedService(allServices);

        return (
            <div className='dashboard'>
                <ScrollToService
                    services={allServices}
                    isMobile={isPhone}
                    plusLoaded={plusLoaded}
                    activated={activatedService}
                />
                {!hideSubscriptions && (plus.allowed || plus.enabled) && <Plus isPhone={isPhone} innerCN={innerCN} />}
                <div className='dashboard__wrapper'>
                    <div className={innerCN}>
                        {!person.isSocialchik && !invalidCloud && (
                            <Cloud data={diskData} items={this.cloudItems} links={navigation.disk} />
                        )}
                        {isMusicVisible && <Music data={musicData} links={navigation.music} />}
                        <Afisha
                            isPhone={isPhone}
                            login={person.login}
                            links={navigation.afisha}
                            data={{
                                purchases: afishaData,
                                favorites: favAfishaData
                            }}
                        />
                        <Market
                            isPhone={isPhone}
                            links={navigation.market}
                            data={{
                                favorites: favMarketData
                            }}
                        />
                        <Collections
                            isPhone={isPhone}
                            publicId={publicId.id}
                            data={collectionsData}
                            links={navigation.collections}
                        />
                        <MapsBookmarks isPhone={isPhone} links={navigation.maps} data={mapsBookmarksData} />
                    </div>
                </div>
            </div>
        );
    }
}

Dashboard.propTypes = {
    dispatch: PropTypes.func,
    publicId: PropTypes.object,
    person: PropTypes.object,
    common: PropTypes.object,
    settings: PropTypes.object,
    dashboard: PropTypes.object
};
