import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {push} from 'react-router-redux';

import {Link} from '@components/Link';
import {Modal} from '@components/Modal';

import {setEditMode} from '../../common/actions';
import {getDevicesList} from './index';
import {DevicesList} from './devices_list.jsx';
import {
    showDevicesWithSameName,
    showDisablingTokens,
    updateDevicesListToDisable,
    updateListToDisable,
    updateTokenTab
} from '@blocks/morda/devices/actions';

class Devices extends Component {
    constructor(props) {
        super(props);

        this.onLinkClick = this.onLinkClick.bind(this);
        this.showDevicesList = this.showDevicesList.bind(this);
    }

    showDevicesList() {
        const {dispatch} = this.props;

        dispatch(setEditMode('devices-list'));
    }

    onLinkClick() {
        const {settings, dispatch} = this.props;
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;

        dispatch(getDevicesList());
        if (isPhone) {
            dispatch(push('/profile/devices'));
        } else {
            this.showDevicesList();
        }
    }

    cancelDisabling = () => {
        const {dispatch, devices} = this.props;
        const {deviceTokens} = devices.tokens;

        if (deviceTokens) {
            dispatch(updateListToDisable());
            dispatch(showDisablingTokens({show: false, deviceId: '', deviceName: ''}));
            dispatch(updateDevicesListToDisable([]));
            dispatch(showDevicesWithSameName());
            dispatch(updateTokenTab());
        }
    };

    closeModal = () => {
        const {dispatch} = this.props;

        dispatch(setEditMode(null));
        this.cancelDisabling();
    };

    render() {
        const {common, settings, devices, havePassword, dispatch} = this.props;
        const edit = common.edit;

        return (
            <div>
                <div className='p-control-link p-control-link__bold' data-t='access-widget:devices'>
                    <Link pseudo={true} onClick={this.onLinkClick}>
                        {i18n('Profile.devices.link-text')}
                    </Link>
                </div>
                <Modal
                    visible={edit === 'devices-list'}
                    zIndexGroupLevel={-0.5}
                    onCloserClick={this.closeModal}
                    dataT='devices'
                >
                    <DevicesList
                        modal={true}
                        devices={devices}
                        dispatch={dispatch}
                        settings={settings}
                        havePassword={havePassword}
                    />
                </Modal>
            </div>
        );
    }
}

function mapStateToProps(state) {
    const {common, settings, devices, person = {}} = state;
    const {havePassword} = person;

    return {
        common,
        settings,
        devices,
        havePassword
    };
}

const WrappedDevices = connect(mapStateToProps)(Devices);

export {WrappedDevices as Devices};

Devices.propTypes = {
    dispatch: PropTypes.func.isRequired,
    common: PropTypes.object,
    devices: PropTypes.object.isRequired,
    settings: PropTypes.object.isRequired,
    havePassword: PropTypes.object.isRequired
};
