import React, {Component} from 'react';
import PropTypes from 'prop-types';

import {Spin} from '@components/Spin';
import {Checkbox} from '@components/Checkbox';
import {Button} from '@components/Button';

import {showRegPopup} from '@blocks/common/actions';
import {
    updateTokenTab,
    showDisablingTokens,
    updateListToDisable,
    updateDevicesListToDisable,
    showDevicesWithSameName
} from './actions';
import {
    revokeDeviceToken,
    disableDeviceById,
    disableDevicesGroup,
    getDevicesList,
    findDevicesWithSameName,
    getDeviceGroupIds
} from './index';
import utils from '@blocks/utils';
import classnames from 'classnames';
import {LINK_CLASS_NAME} from '@blocks/morda/components/link/link.jsx';

export class DevicesList extends Component {
    constructor(props) {
        super(props);

        this.hasPassword = this.props.havePassword || (this.props.person && this.props.person.havePassword);

        this.showInfo = this.showInfo.bind(this);
        this.revokeToken = this.revokeToken.bind(this);
        this.getDate = this.getDate.bind(this);
        this.disableDevice = this.disableDevice.bind(this);
        this.disableDeviceGroup = this.disableDeviceGroup.bind(this);
        this.showDisablingInfo = this.showDisablingInfo.bind(this);
        this.cancelDisabling = this.cancelDisabling.bind(this);
        this.showDubles = this.showDubles.bind(this);
        this.setDisableMessageText = this.setDisableMessageText.bind(this);

        this.devicesContentRef = React.createRef();
    }

    componentDidMount() {
        const {dispatch} = this.props;
        const isStandAlonePage = this.props.common && this.props.common.currentPage !== '/profile';

        this.cancelDisabling();
        // invokes for single page /profile/devices
        if (isStandAlonePage) {
            dispatch(getDevicesList());
        } else {
            if (this.devicesContentRef.current && this.devicesContentRef.current.focus) {
                this.devicesContentRef.current.focus();
            }
        }
    }

    showInfo(e) {
        const {dispatch} = this.props;
        const id = e.currentTarget.getAttribute('data-tokenInfo');
        const type = e.currentTarget.getAttribute('data-tokenType');

        dispatch(updateTokenTab(id, type));
    }

    preventFold(e) {
        e.stopPropagation();
    }

    getDate(timestamp) {
        if (!timestamp) {
            return '';
        }

        return utils
            .dateToText(new Date(timestamp * 1000), {
                year: true,
                time: true
            })
            .toLowerCase();
    }

    revokeToken(e) {
        e.stopPropagation();
        if (!this.hasPassword) {
            this.props.dispatch(showRegPopup(true));
            return;
        }
        const tokenId = e.currentTarget.getAttribute('data-tokenId');
        const tokenType = e.currentTarget.getAttribute('data-tokenType');

        this.props.dispatch(revokeDeviceToken(tokenId, tokenType));
    }

    disableDevice(e) {
        const deviceId =
            (e.currentTarget && e.currentTarget.getAttribute('data-deviceId')) || this.props.devices.disableDeviceId;
        const hasXtoken = e.currentTarget && e.currentTarget.getAttribute('data-hasXtoken');
        const groupToDelete = this.props.devices.devicesToDelete;

        if (hasXtoken === undefined || hasXtoken === null) {
            if (groupToDelete.length > 0) {
                this.props.dispatch(disableDevicesGroup(groupToDelete, this.hasPassword));
            } else {
                this.props.dispatch(disableDeviceById(deviceId, this.hasPassword));
            }
        } else {
            this.showDisablingInfo(deviceId);
        }
    }

    disableDeviceGroup(e) {
        const deviceId =
            (e.currentTarget && e.currentTarget.getAttribute('data-deviceId')) || this.props.devices.disableDeviceId;

        this.props.dispatch(getDeviceGroupIds(deviceId));
    }

    showDisablingInfo(deviceId) {
        const deviceName = this.props.devices.tokens.deviceTokens.filter((el) => el.deviceId === deviceId)[0]
            .deviceName;
        const disableInfo = {
            show: true,
            deviceId,
            deviceName
        };

        this.props.dispatch(showDisablingTokens(disableInfo));
        this.props.dispatch(updateTokenTab());
    }

    cancelDisabling() {
        const {dispatch, devices} = this.props;
        const {deviceTokens} = devices.tokens;

        if (deviceTokens) {
            dispatch(updateListToDisable());
            dispatch(showDisablingTokens({show: false, deviceId: '', deviceName: ''}));
            dispatch(updateDevicesListToDisable([]));
            dispatch(showDevicesWithSameName());
            dispatch(updateTokenTab());
        }
    }

    setDisableMessageText(devicesToDelete, deviceName) {
        let text = i18n('Profile.devices.disable-text_nodevice');

        if (deviceName) {
            if (devicesToDelete.length > 0) {
                text = i18n('Profile.devices.disable-group-text').replace('%s', deviceName);
            } else {
                text = i18n('Profile.devices.disable-text').replace('%s', deviceName);
            }
        }

        return text;
    }

    showDubles(e) {
        const deviceId = e.currentTarget && e.currentTarget.getAttribute('data-deviceId');

        this.props.dispatch(findDevicesWithSameName(deviceId));
    }

    render() {
        const {devices, dispatch, settings = {}} = this.props;
        const {ua = {}} = settings;
        const {isMobile} = ua;
        const {deviceTokens, otherTokens, isYandexTokens, tokensPerDevice} = devices.tokens;
        const {showDisablingList, devicesToDelete} = devices;
        const disablingId = devices.disableDeviceId;
        const hasNoAnyTokens =
            deviceTokens && deviceTokens.length === 0 && otherTokens.length === 0 && isYandexTokens.length === 0;
        const descrText = hasNoAnyTokens ? i18n('Profile.device.empty-list') : i18n('Profile.devices.description');

        return (
            <div className={classnames('p-control-form manage-devices')} tabIndex='0' ref={this.devicesContentRef}>
                <span className='personal-spinner'>
                    <Spin
                        size='s'
                        progress={deviceTokens === null || otherTokens === null || isYandexTokens === null}
                    />
                </span>
                {showDisablingList ? (
                    <div>
                        <h2 className='section-title'>{i18n('Profile.devices.disable-title')}</h2>
                        <div className='section-info'>
                            {this.setDisableMessageText(devicesToDelete, devices.deviceName)}
                        </div>
                        <div className='devices-list_content'>
                            <div className='info-list'>
                                {deviceTokens.map(
                                    (device) =>
                                        Boolean(
                                            device.deviceId === disablingId &&
                                                !device.hasXtoken &&
                                                isYandexTokens.length > 0
                                        ) && (
                                            <div key={device.deviceId}>
                                                <div className='manage-devices_text'>
                                                    {i18n('Profile.devices.disable-text_tokens')}
                                                </div>

                                                <OtherTokens
                                                    dispatch={dispatch}
                                                    tokens={isYandexTokens}
                                                    tokensType='isYandexTokens'
                                                    showCheckbox='yes'
                                                    getDate={this.getDate}
                                                    showInfo={this.showInfo}
                                                    revokeToken={this.revokeToken}
                                                    preventFold={this.preventFold}
                                                />
                                            </div>
                                        )
                                )}
                                <div className='controls-wrapper'>
                                    <div className='right-aligned'>
                                        <Button
                                            view='action'
                                            size='l'
                                            type='button'
                                            onClick={this.disableDevice}
                                            width={isMobile ? 'max' : ''}
                                        >
                                            {i18n('Profile.device.disable-confirm')}
                                        </Button>
                                    </div>
                                    <div className='left-aligned'>
                                        <Button
                                            view='pseudo'
                                            size='l'
                                            type='button'
                                            onClick={this.cancelDisabling}
                                            width={isMobile ? 'max' : ''}
                                        >
                                            {i18n('_AUTH_.common.cancel')}
                                        </Button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                ) : (
                    <div>
                        <h2 className='section-title'>{i18n('Profile.devices.title')}</h2>
                        <div className='section-info'>{descrText}</div>
                        <div className='devices-list_content'>
                            <div className='info-list'>
                                <div className='devices-section'>
                                    {Boolean(deviceTokens && deviceTokens.length > 0) && (
                                        <div className='device-section-wrapper'>
                                            <div className='section-name'>{i18n('Profile.devices.group-title')}</div>
                                            <div className='section-name_descr'>
                                                {i18n('Profile.devices.group-description')}
                                            </div>
                                        </div>
                                    )}
                                    {Boolean(deviceTokens && deviceTokens.length > 0) &&
                                        deviceTokens.map((device) => (
                                            <div
                                                key={device.deviceId}
                                                className={classnames('device-item', {
                                                    is_duble: device.duble,
                                                    show: device.showDuble
                                                })}
                                            >
                                                <div
                                                    className={classnames('device-name-wrapper', {
                                                        no_tokens: device.tokens && device.tokens.length === 0
                                                    })}
                                                >
                                                    <div className='device-name'>
                                                        {device.deviceName
                                                            ? device.deviceName
                                                            : i18n('Profile.devices.default-device')}
                                                    </div>
                                                    <div>
                                                        <div
                                                            className={`${LINK_CLASS_NAME} disable-device`}
                                                            data-deviceId={device.deviceId}
                                                            data-hasXtoken={device.hasXtoken}
                                                            onClick={this.disableDevice}
                                                            role='link'
                                                            tabIndex='0'
                                                        >
                                                            {i18n('Profile.devices.disable-device')}
                                                        </div>

                                                        {Boolean(device.hasDubles && !device.duble) && (
                                                            <div
                                                                className={`${LINK_CLASS_NAME} disable-device`}
                                                                data-deviceId={device.deviceId}
                                                                onClick={this.disableDeviceGroup}
                                                                role='link'
                                                                tabIndex='0'
                                                            >
                                                                {i18n('Profile.devices.disable-by-name').replace(
                                                                    '%s',
                                                                    device.deviceName
                                                                )}
                                                            </div>
                                                        )}
                                                    </div>

                                                    {Boolean(
                                                        tokensPerDevice[device.deviceId] &&
                                                            tokensPerDevice[device.deviceId][0]
                                                    ) && (
                                                        <span className='device-token_last-auth'>
                                                            {i18n('Profile.lastauth.entry').replace(
                                                                '%1',
                                                                this.getDate(tokensPerDevice[device.deviceId][0])
                                                            )}
                                                        </span>
                                                    )}

                                                    {Boolean(device.hasDubles && !device.duble) && (
                                                        <div>
                                                            <div>{i18n('Profile.devices.same-names')}</div>
                                                            <button
                                                                className={`manage-devices-toggle ${LINK_CLASS_NAME}`}
                                                                onClick={this.showDubles}
                                                                data-deviceId={device.deviceId}
                                                            >
                                                                {devices.deviceName === device.deviceName
                                                                    ? i18n('Profile.devices.hide')
                                                                    : i18n('Profile.devices.show')}
                                                            </button>
                                                        </div>
                                                    )}
                                                </div>
                                                <DeviceTokens
                                                    tokens={device.tokens}
                                                    getDate={this.getDate}
                                                    showInfo={this.showInfo}
                                                    revokeToken={this.revokeToken}
                                                    preventFold={this.preventFold}
                                                />
                                            </div>
                                        ))}

                                    {Boolean(isYandexTokens && isYandexTokens.length > 0) && (
                                        <div className='device-item'>
                                            <div className='device-name-wrapper'>
                                                <div className='device-name'>{i18n('Profile.devices.yandex')}</div>
                                                <div className='device-name_descr'>
                                                    {i18n('Profile.devices.yandex-description')}
                                                </div>
                                            </div>
                                            <OtherTokens
                                                tokens={isYandexTokens}
                                                tokensType='isYandexTokens'
                                                getDate={this.getDate}
                                                showInfo={this.showInfo}
                                                revokeToken={this.revokeToken}
                                                preventFold={this.preventFold}
                                            />
                                        </div>
                                    )}
                                </div>

                                {Boolean(otherTokens && otherTokens.length > 0) && (
                                    <div>
                                        <div className='device-name-wrapper'>
                                            <div className='device-name'>{i18n('Profile.devices.other')}</div>
                                            <div className='device-name_descr'>
                                                {i18n('Profile.devices.other-description')}
                                            </div>
                                        </div>
                                        <OtherTokens
                                            tokens={otherTokens}
                                            tokensType='otherTokens'
                                            getDate={this.getDate}
                                            showInfo={this.showInfo}
                                            revokeToken={this.revokeToken}
                                            preventFold={this.preventFold}
                                        />
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                )}
            </div>
        );
    }
}

class DeviceTokens extends Component {
    render() {
        const {tokens, getDate, showInfo, revokeToken} = this.props;

        return (
            <div className='info-list-items'>
                {tokens.map((token) => (
                    <div
                        className={classnames('info-list_item', {active: token.tabOpened})}
                        key={token.key}
                        data-tokenInfo={token.tokenId}
                        data-tokenType='deviceTokens'
                        onClick={showInfo}
                    >
                        <span className='toggler' />
                        <div className='client-info'>
                            <ClientIcon
                                iconUrl={token.client.iconUrl}
                                name={token.client.title}
                                isYandex={token.client.isYandex}
                            />
                            <span className='client-name'>{token.client.title}</span>
                            <div className='last-auth'>{getDate(token.issueTime)}</div>
                            <div className='extended-info' onClick={this.props.preventFold}>
                                <div className='token-data'>
                                    {i18n('Profile.access.widget-tokenId')} {token.client.id} <br />
                                    {i18n('Profile.access.widget-created-date')} {getDate(token.issueTime)}
                                    <br />
                                    {i18n('Profile.device.token-source')}{' '}
                                    {token.client.isYandex
                                        ? i18n('_AUTH_.common.yandex')
                                        : i18n('Profile.devices.another')}
                                    <div
                                        className={LINK_CLASS_NAME}
                                        data-tokenId={token.tokenId}
                                        data-tokenType='deviceTokens'
                                        onClick={revokeToken}
                                    >
                                        {i18n('Profile.common.delete')}
                                    </div>
                                </div>
                                <dl className='scopes-list'>
                                    {token.scopes.map((devicesItem, itemIndex) => (
                                        <span key={itemIndex}>
                                            <dt>{devicesItem.scopeTitle}</dt>
                                            {devicesItem.scopesList.map((devicesItemlist, index) => (
                                                <dd key={index}>{devicesItemlist}</dd>
                                            ))}
                                        </span>
                                    ))}
                                </dl>
                            </div>
                        </div>
                    </div>
                ))}
            </div>
        );
    }
}

class OtherTokens extends Component {
    render() {
        const {tokens, getDate, tokensType, showInfo, revokeToken, showCheckbox, dispatch} = this.props;

        return (
            <div className={classnames('info-list-items', {checkboxes: showCheckbox})}>
                {tokens.map((token) => (
                    <div
                        className={classnames('info-list_item', {active: token.tabOpened})}
                        key={token.tokenId}
                        data-tokenInfo={token.tokenId}
                        data-tokenType={tokensType}
                        onClick={showInfo}
                    >
                        <span className='toggler' />
                        <div className='client-info'>
                            {Boolean(showCheckbox) && (
                                <OtherTokensCheckbox token={token} dispatch={dispatch} tokensType={tokensType} />
                            )}

                            <ClientIcon
                                iconUrl={token.client.iconUrl}
                                name={token.client.title}
                                isYandex={token.client.isYandex}
                            />
                            <span className='client-name'>{token.client.title}</span>
                            <div className='last-auth'>{getDate(token.issueTime)}</div>
                            <div className='extended-info' onClick={this.props.preventFold}>
                                <div className='token-data'>
                                    {i18n('Profile.access.widget-tokenId')} {token.client.id} <br />
                                    {i18n('Profile.access.widget-created-date')} {getDate(token.issueTime)}
                                    <div
                                        className={LINK_CLASS_NAME}
                                        data-tokenId={token.tokenId}
                                        data-tokenType={tokensType}
                                        onClick={revokeToken}
                                    >
                                        {i18n('Profile.common.delete')}
                                    </div>
                                </div>
                                <dl className='scopes-list'>
                                    {token.scopes.map((item, itemIndex) => (
                                        <span key={itemIndex}>
                                            <dt>{item.scopeTitle}</dt>
                                            {item.scopesList.map((itemlist, index) => (
                                                <dd key={index}>{itemlist}</dd>
                                            ))}
                                        </span>
                                    ))}
                                </dl>
                            </div>
                        </div>
                    </div>
                ))}
            </div>
        );
    }
}

class OtherTokensCheckbox extends Component {
    constructor(props) {
        super(props);

        this.addTokensToDelete = this.addTokensToDelete.bind(this);

        this.state = {
            checked: false
        };
    }

    addTokensToDelete() {
        const state = !this.state.checked ? 'checked' : 'unchecked';
        const id = this.props.token.tokenId;

        this.setState((prevProps) => ({checked: !prevProps.checked}));
        this.props.dispatch(updateListToDisable({id, state, type: this.props.tokensType}));
    }

    render() {
        const {tokenId} = this.props.token;

        return (
            <div className='disabling-mark'>
                <Checkbox
                    value={tokenId}
                    size='m'
                    checked={this.state.checked}
                    onChange={this.addTokensToDelete}
                    name='token'
                />
            </div>
        );
    }
}

class ClientIcon extends Component {
    constructor(props) {
        super(props);

        this.returnFirstLetter = this.returnFirstLetter.bind(this);
    }

    shouldComponentUpdate() {
        return false;
    }

    returnFirstLetter(str) {
        return str.charAt(0);
    }

    render() {
        const {iconUrl, name, isYandex} = this.props;
        const bgStyle = {
            backgroundImage: `url(${iconUrl})`
        };
        const styleClasses = classnames('client-icon', {'is-yandex': isYandex});

        return (
            <span>
                {(() => {
                    if (iconUrl) {
                        return <span className={styleClasses} style={bgStyle} />;
                    }

                    if (name) {
                        return <span className={styleClasses}>{this.returnFirstLetter(name)}</span>;
                    }

                    return <span className='client-icon default-icon' />;
                })()}
            </span>
        );
    }
}

DevicesList.propTypes = {
    dispatch: PropTypes.func.isRequired,
    devices: PropTypes.shape({
        tokens: PropTypes.object.isRequired,
        deviceName: PropTypes.string,
        showDisablingList: PropTypes.bool,
        otherTokensToDelete: PropTypes.array,
        disableDeviceId: PropTypes.string,
        devicesToDelete: PropTypes.array
    }),
    access: PropTypes.object,
    modal: PropTypes.bool,
    common: PropTypes.object,
    settings: PropTypes.object,
    person: PropTypes.object,
    havePassword: PropTypes.bool
};

DeviceTokens.propTypes = {
    getDate: PropTypes.func.isRequired,
    preventFold: PropTypes.func,
    revokeToken: PropTypes.func,
    showInfo: PropTypes.func,
    tokens: PropTypes.array
};

OtherTokens.propTypes = {
    getDate: PropTypes.func.isRequired,
    preventFold: PropTypes.func,
    revokeToken: PropTypes.func,
    showInfo: PropTypes.func,
    tokens: PropTypes.array,
    tokensType: PropTypes.string,
    showCheckbox: PropTypes.string,
    dispatch: PropTypes.func
};

OtherTokensCheckbox.propTypes = {
    token: PropTypes.object.isRequired,
    tokensType: PropTypes.string,
    dispatch: PropTypes.func
};

ClientIcon.propTypes = {
    isYandex: PropTypes.bool,
    iconUrl: PropTypes.string,
    name: PropTypes.string
};
