import React, {useState, useCallback} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';

import {Button} from '@yandex-id/components';
import {useMatchMedia} from '@yandex-id/components/utils/hooks';

import {DocumentsImageViewer} from '../ImageViewer/Documents-ImageViewer';

import './Documents-Explorer.styl';

const cnDocumentsExplorer = cn('DocumentsExplorer');

export const DocumentsExplorer = (props) => {
    const {className, items, onClose, onCopy} = props;

    const [isCopyProcessing, setIsCopyProcessing] = useState(false);

    const [images, setImages] = useState(
        items.reduce((acc, item) => {
            if (item.preview) {
                acc.push({
                    src: item.preview,
                    name: item.name,
                    path: item.path,
                    selected: false
                });
            }

            return acc;
        }, [])
    );

    const [currentImage, setCurentImage] = useState();
    const [isImageViewerVisible, setIsImageViewerVisible] = useState(false);

    const handleImage = (index) => () => {
        setCurentImage(index);
        setIsImageViewerVisible(true);
    };

    const onCloseImageViewer = useCallback(() => setIsImageViewerVisible(false), []);

    const toogleSelected = useCallback(
        (path) => () => {
            const index = images.findIndex((image) => {
                if (image.path === path) {
                    return true;
                }
            });

            if (index !== -1) {
                images[index] = {
                    ...images[index],
                    selected: !images[index].selected
                };

                setImages([...images]);
            }
        },
        [images]
    );

    const onClickCopy = useCallback(async () => {
        const copy = images.reduce((acc, image, index) => {
            if (image.selected) {
                acc.push(index);
            }

            return acc;
        }, []);

        if (copy.length > 0) {
            setIsCopyProcessing(true);
            await onCopy(copy).then(() => setIsCopyProcessing(false));
        }
    }, [images, onCopy]);

    const renderImages = images.map((image, index) => {
        const {src, name, path, selected} = image;

        return (
            <div key={index} className={cnDocumentsExplorer('ImageTile')}>
                <div className={cnDocumentsExplorer('SelectionTile')} onClick={toogleSelected(path)}>
                    <div
                        className={cnDocumentsExplorer('Selection', {
                            selected
                        })}
                    />
                </div>
                <div className={cnDocumentsExplorer('ImageWrapper')}>
                    <img className={cnDocumentsExplorer('Image')} src={src} onClick={handleImage(index)} />
                </div>
                <div className={cnDocumentsExplorer('Name')}>{name}</div>
            </div>
        );
    });

    if (renderImages.length % 2 !== 0) {
        renderImages.push(
            <div key={renderImages.length} className={cnDocumentsExplorer('ImageTile', {empty: true})} />
        );
    }

    const isImagesSelected = Boolean(images.filter(({selected}) => selected).length);
    const closeButton = useMatchMedia('(min-width: 768px)') ? (
        <div className={cnDocumentsExplorer('Close')} onClick={onClose} />
    ) : null;

    return (
        <div className={cnDocumentsExplorer('', [className])}>
            <div className={cnDocumentsExplorer('Header')}>
                <div className={cnDocumentsExplorer('Title')}>{i18n('Profile2.documents.info.view.finded')}</div>
                {closeButton}
            </div>
            <div className={cnDocumentsExplorer('Content')}>{renderImages}</div>
            <div className={cnDocumentsExplorer('Footer')}>
                <Button
                    className={cnDocumentsExplorer('Add')}
                    view='action'
                    width='max'
                    pin='circle-circle'
                    size='l'
                    disabled={!isImagesSelected}
                    progress={isCopyProcessing}
                    onClick={onClickCopy}
                >
                    {isImagesSelected
                        ? i18n('Profile2.documents.info.view.add')
                        : i18n('Profile2.documents.info.view.select')}
                </Button>
            </div>
            <DocumentsImageViewer
                className={cnDocumentsExplorer('Viewer')}
                isVisible={isImageViewerVisible}
                images={items.filter((item) => item.preview)}
                startImageIndex={currentImage}
                onClose={onCloseImageViewer}
            />
        </div>
    );
};

DocumentsExplorer.propTypes = {
    className: PropTypes.string,
    items: PropTypes.array.isRequired,
    onClose: PropTypes.func,
    onCopy: PropTypes.func
};
