import React, {useRef, useCallback, useState} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import ImageGallery from 'react-image-gallery';

import {Dialog} from '@yandex-id/components/components/Dialog';

import {DeleteConfirmation} from '../DeleteConfirmation/Documents-DeleteConfirmation';

import 'react-image-gallery/styles/css/image-gallery.css';
import './Documents-ImageViewer.styl';

const cnDocumentsImageViewer = cn('DocumentsImageViewer');

export const DocumentsImageViewer = (props) => {
    const {className, images, startImageIndex = 0, isVisible, onClose, onDeleteImage} = props;

    const [isDeleteVisible, setIsDeleteVisible] = useState(false);

    const onOpenDelete = useCallback(() => setIsDeleteVisible(true), []);
    const onCloseDelete = useCallback(() => setIsDeleteVisible(false), []);

    const galleryImages = images.map((image) => {
        return {
            originalClass: cnDocumentsImageViewer('Image'),
            original: image.original_url || image.preview
        };
    });

    const onDelete = useCallback(() => {
        onCloseDelete();

        const index = galleryRef.current.state.currentIndex;
        const itemsCount = galleryRef.current.props.items.length;

        if (itemsCount - 1 === 0) {
            onClose();
        } else if (itemsCount - 1 === index) {
            galleryRef.current.slideLeft();
        } else {
            galleryRef.current.slideRight();
        }

        onDeleteImage(index);
    }, [onClose, onDeleteImage, onCloseDelete]);

    const onDownload = useCallback(() => {
        const index = galleryRef.current.state.currentIndex;
        const image = images[index];
        const element = document.createElement('a');

        element.setAttribute('href', image.original_url ? `${image.original_url}?download` : image.file);
        element.setAttribute('download', 'download');
        document.body.appendChild(element);
        element.click();
        document.body.removeChild(element);
    }, [images]);

    const galleryRef = useRef();

    return (
        <Dialog
            className={cnDocumentsImageViewer('', [className])}
            classNameContent={cnDocumentsImageViewer('Content')}
            theme={null}
            type='modal'
            visible={isVisible}
            zIndexGroupLevel={20}
            keepMounted={false}
            onClose={onClose}
        >
            <div className={cnDocumentsImageViewer('Nav', {close: true})} onClick={onClose} />
            {onDeleteImage && <div className={cnDocumentsImageViewer('Nav', {delete: true})} onClick={onOpenDelete} />}
            <div
                className={cnDocumentsImageViewer('Nav', {download: true, center: !onDeleteImage})}
                onClick={onDownload}
            />
            <ImageGallery
                ref={galleryRef}
                items={galleryImages}
                showThumbnails={false}
                showPlayButton={false}
                startIndex={startImageIndex}
                showFullscreenButton={false}
                stopPropagation={true}
                renderLeftNav={(click, disabled) => (
                    <div
                        className={cnDocumentsImageViewer('Nav', {left: true})}
                        onClick={click}
                        style={{display: disabled ? 'none' : 'block'}}
                    />
                )}
                renderRightNav={(click, disabled) => (
                    <div
                        className={cnDocumentsImageViewer('Nav', {right: true})}
                        onClick={click}
                        style={{display: disabled ? 'none' : 'block'}}
                    />
                )}
            />
            {onDeleteImage && (
                <DeleteConfirmation isVisible={isDeleteVisible} onAbort={onCloseDelete} onConfirm={onDelete} />
            )}
        </Dialog>
    );
};

DocumentsImageViewer.propTypes = {
    className: PropTypes.string,
    images: PropTypes.array.isRequired,
    startImageIndex: PropTypes.number,
    isVisible: PropTypes.bool.isRequired,
    onClose: PropTypes.func.isRequired,
    onDeleteImage: PropTypes.func
};
