import React, {useState, useEffect, useCallback} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';

import {Document} from '@yandex-id/components';
import {Dialog} from '@yandex-id/components/components/Dialog';

import {DocumentsInfo} from './Info/Documents-Info';

import {DocumentsExplorer} from '../../Explorer/Documents-Explorer';

import {documentMapper} from '../../Documents.utils';

import {
    getAllSearchDocuments,
    copyDocument,
    getAllDocumentsByType,
    uploadDocumentByType,
    deleteDocumentById
} from '../../api';

import './Documents-ListItem.styl';

const cnListItem = cn('ListItem');

export const DocumentsListItem = (props) => {
    const {userUid, isRegisterRequired, docTitle, docId, scope} = props;

    const [isLoading, setIsLoading] = useState(!isRegisterRequired);
    const [isLoaded, setIsLoaded] = useState(false);
    const [shouldLoadData, setShoudLoadData] = useState(!isRegisterRequired && !isLoaded);

    const [documents, setDocuments] = useState([]);
    const [documentsDiskSearch, setDocumentsDiskSearch] = useState([]);

    const [isUploadingPhoto, setIsUploadingPhoto] = useState(false);
    const [isInfoVisible, setIsInfoVisible] = useState(false);
    const [isDiskViewerVisible, setIsDiskViewerVisible] = useState(false);

    const openDiskViewer = () => setIsDiskViewerVisible(true);
    const closeDiskViewer = () => setIsDiskViewerVisible(false);

    const onOpenInfo = () => setIsInfoVisible(true);
    const onCloseInfo = useCallback(() => {
        setIsInfoVisible(false);
    }, []);

    const onDeleteDocument = useCallback(
        async (index) => {
            if (typeof index !== 'number') {
                return;
            }

            const document = documents[index];

            const tempDocs = documents;

            tempDocs.splice(index, 1);

            setDocuments([...tempDocs]);

            await deleteDocumentById(userUid, document.id);
        },
        [userUid, documents]
    );

    const onCopyToDocument = useCallback(
        async (indexes = []) => {
            if (indexes.length === 0) {
                return;
            }

            try {
                await Promise.all(
                    indexes.map((itemIndex) =>
                        copyDocument(userUid, documentsDiskSearch[itemIndex].path, documentMapper[docId])
                    )
                ).then((results) => setDocuments([...documents, ...results]));
            } finally {
                closeDiskViewer();
            }
        },
        [documentsDiskSearch, userUid, documents, docId]
    );

    const onUploadDocument = useCallback(
        async (files) => {
            try {
                setIsUploadingPhoto(true);

                const uploadRequests = await Object.values(files).reduce(async (acc, file) => {
                    const temp = await acc;
                    const uploadDocumentData = await uploadDocumentByType(userUid, documentMapper[docId], file);

                    temp.push(uploadDocumentData);

                    return Promise.resolve(temp);
                }, Promise.resolve([]));

                await Promise.all(uploadRequests).then((result) => {
                    setDocuments([...documents, ...result]);
                });
            } finally {
                setIsUploadingPhoto(false);
            }
        },
        [docId, userUid, documents]
    );

    useEffect(() => {
        if (shouldLoadData) {
            const loadData = async () => {
                setShoudLoadData(false);

                try {
                    const results = await Promise.all([
                        getAllDocumentsByType(userUid, documentMapper[docId]),
                        getAllSearchDocuments(docTitle)
                    ]);

                    setDocuments(results[0]);
                    setDocumentsDiskSearch(results[1].filter((result) => result.preview));

                    setIsLoaded(true);
                } catch (error) {
                    console.error(error);
                } finally {
                    setIsLoading(false);
                }
            };

            loadData();
        }
    }, [shouldLoadData, userUid, docId, docTitle]);

    if (isRegisterRequired) {
        return (
            <Document
                className={cnListItem('Document')}
                data={{
                    id: docId,
                    title: docTitle,
                    isActive: false
                }}
            />
        );
    }

    const hasSearchItems = Boolean(documentsDiskSearch.length);

    return (
        <>
            <Document
                className={cnListItem('Document')}
                data={{
                    id: docId,
                    title: docTitle,
                    isActive: true
                }}
                onClick={onOpenInfo}
            />
            <Dialog
                type='auto'
                modal={{
                    classNameContent: cnListItem('Info', {desktop: true}),
                    visible: isInfoVisible,
                    zIndexGroupLevel: 20,
                    keepMounted: false,
                    onClose: onCloseInfo
                }}
                drawer={{
                    classNameContent: cnListItem('Info', {mobile: true}),
                    keepMounted: false,
                    visible: isInfoVisible,
                    scope,
                    onClose: onCloseInfo
                }}
            >
                <DocumentsInfo
                    isLoading={isLoading}
                    docTitle={docTitle}
                    searchItems={documentsDiskSearch}
                    documents={documents}
                    isPhotoUploading={isUploadingPhoto}
                    onOpenDiskViewer={openDiskViewer}
                    onDeleteDocument={onDeleteDocument}
                    onUploadDocument={onUploadDocument}
                />
            </Dialog>
            {!isLoading && hasSearchItems && (
                <>
                    <Dialog
                        type='auto'
                        modal={{
                            classNameContent: cnListItem('DiskTile', {desktop: true}),
                            visible: isDiskViewerVisible,
                            zIndexGroupLevel: 20,
                            keepMounted: false,
                            onClose: closeDiskViewer
                        }}
                        drawer={{
                            classNameContent: cnListItem('DiskTile', {mobile: true}),
                            keepMounted: false,
                            visible: isDiskViewerVisible,
                            scope,
                            onClose: closeDiskViewer
                        }}
                    >
                        <DocumentsExplorer
                            className={cnListItem('DiskViewer')}
                            items={documentsDiskSearch}
                            onClose={closeDiskViewer}
                            onCopy={onCopyToDocument}
                        />
                    </Dialog>
                </>
            )}
        </>
    );
};

DocumentsListItem.propTypes = {
    userUid: PropTypes.number.isRequired,
    isRegisterRequired: PropTypes.bool.isRequired,
    docId: PropTypes.string,
    docTitle: PropTypes.string,
    scope: PropTypes.object
};
