import React, {useState, useCallback} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';

import {Button, Icon, Spin} from '@yandex-id/components';

import './Documents-Info.styl';
import {DocumentsImageViewer} from '../../../ImageViewer/Documents-ImageViewer';

const cnDocumentsInfo = cn('Info');

export const DocumentsInfo = (props) => {
    const {
        isLoading,
        docTitle,
        searchItems = [],
        documents = [],
        onOpenDiskViewer,
        onDeleteDocument,
        onUploadDocument,
        isPhotoUploading
    } = props;

    const [currentImage, setCurentImage] = useState();
    const [isImageViewerVisible, setIsImageViewerVisible] = useState(false);

    const onCloseImageViewer = useCallback(() => setIsImageViewerVisible(false), []);

    const handleImage = (index) => () => {
        setCurentImage(index);
        setIsImageViewerVisible(true);
    };

    const images = documents.map((item) => item.preview_url);

    const imagesPreview = images.map((image, index) => (
        <img key={index} className={cnDocumentsInfo('Photo')} src={image} onClick={handleImage(index)} />
    ));

    const i18nAddPhoto = i18n('Profile2.documents.info.addPhoto').split('{break}');
    const i18nSearchDescription = (searchItems.length === 1
        ? i18n('Profile2.documents.info.searchDescription.one')
        : i18n('Profile2.documents.info.searchDescription')
    )
        .replace('{count}', String(searchItems.length))
        .split(' ');

    return (
        <div className={cnDocumentsInfo()}>
            <div className={cnDocumentsInfo('Preview', {withSearch: Boolean(searchItems.length)})}>
                <div>
                    <div className={cnDocumentsInfo('Name')}>{docTitle}</div>
                    {documents.length === 0 && (
                        <div className={cnDocumentsInfo('Description')}>
                            {i18n('Profile2.documents.info.description')}
                        </div>
                    )}
                </div>
                <div className={cnDocumentsInfo('Photos')}>
                    {imagesPreview}
                    <input
                        accept='image/jpeg, image/jpg, image/png, image/heic'
                        id='document-uploader'
                        type='file'
                        style={{display: 'none'}}
                        onChange={(e) => onUploadDocument(e.target.files)}
                        multiple={true}
                    />
                    <label htmlFor='document-uploader'>
                        <Button
                            className={cnDocumentsInfo('Add', {
                                empty: documents.length !== 0,
                                isUploading: isPhotoUploading
                            })}
                            view='pseudo'
                            width={
                                Boolean(documents.length) || (Boolean(documents.length) && Boolean(searchItems.lengh))
                                    ? 'max'
                                    : 'auto'
                            }
                            pin='circle-circle'
                            disabled={isPhotoUploading}
                            as='span'
                        >
                            {isPhotoUploading || isLoading ? (
                                <Spin view='default' progress={true} position='center' size='m' />
                            ) : (
                                <>
                                    <Icon glyph='plus' />
                                    <span style={{fontSize: '13px', lineHeight: '16px'}}>
                                        {i18nAddPhoto[0] || ''}
                                        <br />
                                        {i18nAddPhoto.slice(1).join(' ') || ''}
                                    </span>
                                </>
                            )}
                        </Button>
                    </label>
                </div>
            </div>
            {!isLoading && searchItems.length > 0 && (
                <div className={cnDocumentsInfo('Search')}>
                    <p className={cnDocumentsInfo('Title')}>
                        <b>{i18nSearchDescription.slice(0, 2).join(' ') || ''}</b>
                        &nbsp;
                        {i18nSearchDescription.slice(2).join(' ') || ''}
                        &nbsp;
                        {docTitle}
                    </p>
                    <Button
                        className={cnDocumentsInfo('Find')}
                        view='default'
                        pin='circle-circle'
                        size='l'
                        width='max'
                        onClick={onOpenDiskViewer}
                    >
                        <Icon style={{width: '24px'}} glyph='find' />
                        <span style={{marginLeft: '8px'}}>{i18n('Profile2.documents.info.view')}</span>
                    </Button>
                </div>
            )}
            {!isLoading && documents.length > 0 && (
                <DocumentsImageViewer
                    className={cnDocumentsInfo('View')}
                    isVisible={isImageViewerVisible}
                    images={documents}
                    startImageIndex={currentImage}
                    onClose={onCloseImageViewer}
                    onDeleteImage={onDeleteDocument}
                />
            )}
        </div>
    );
};

DocumentsInfo.propTypes = {
    isLoading: PropTypes.bool.isRequired,
    docTitle: PropTypes.string,
    searchItems: PropTypes.array,
    documents: PropTypes.array,
    isPhotoUploading: PropTypes.bool,
    onOpenDiskViewer: PropTypes.func.isRequired,
    onDeleteDocument: PropTypes.func,
    onUploadDocument: PropTypes.func
};
