import {setInitDisk, setDocsFolderName, setDocsState, setScansFolder} from './actions/actions';
import api from '@blocks/api';
import {docsName, docs, docsFolder} from './constants';
import {getFolderDocName, getDocNamesById, getLocalizedDocNameById} from './Documents.utils';

export const initDisk = (lang, folderDoc, dispatch) => {
    // Получаем данные о Диске – нам нужны пути к системным
    return api.request('documents.disk.getMeta').done((result) => {
        // Пытаемся создать системную папку – её может не быть.
        // Если получаем Дубликат, создаём папки дальше.
        const systemScansFolder = result.system_folders.scans;

        return api
            .request('documents.disk.createFolder', {
                folderPath: systemScansFolder
            })
            .done(() => {
                // Пытаемся создать папку с личными док-ами.
                return api
                    .request('documents.disk.createFolder', {
                        folderPath: `${systemScansFolder}/${folderDoc}`
                    })
                    .done(() => {
                        // Делаем мапу с именами док-ов в локале пользователя.
                        // И для каждого дока создаём папку.
                        const raw = docsName.reduce((acc, docName) => {
                            acc[docName] = docs[docName][lang];
                            api.request('documents.disk.createFolder', {
                                folderPath: `${systemScansFolder}/${folderDoc}/${docs[docName][lang]}`
                            })
                                .done(() => {})
                                .fail(() => {});

                            return acc;
                        }, {});

                        const scans = systemScansFolder.substring(systemScansFolder.indexOf('/') + 1);

                        window.localStorage.setItem('scansFolder', scans);
                        // Сохраняем путь папки с док-ами.
                        window.localStorage.setItem('docsFolder', folderDoc);
                        // Сохраняем имена папок в локале юзера
                        window.localStorage.setItem('docs', JSON.stringify(raw));
                        // Проставляем флажок, что заинитили Диск.
                        window.localStorage.setItem('isInited', '1');
                        // Всё тоже самое, проставляем в стейт.
                        dispatch(setScansFolder(scans));
                        dispatch(setDocsFolderName(folderDoc));
                        dispatch(setDocsState(raw));
                        dispatch(setInitDisk(1));
                    });
            });
    });
};

export const initPersonalDocuments = async (lang) => {
    const result = await Promise.resolve(api.request('documents.disk.getMeta'));
    const systemScansFolder = result.system_folders.scans;
    const data = {
        scansFolder: systemScansFolder.substring(systemScansFolder.indexOf('/') + 1)
    };

    await Promise.resolve(
        api.request('documents.disk.createFolder', {
            folderPath: systemScansFolder
        })
    );
    const findPersonalFolderRequests = Object.values(docsFolder).map((docsFolderName) =>
        api.request('documents.disk.getResourceMeta', {
            folderPath: `${systemScansFolder}/${docsFolderName}`
        })
    );
    const results = await Promise.all(findPersonalFolderRequests);

    const info = results.reduce((acc, res) => {
        if (res.path) {
            acc.name = res.name;
        }

        return acc;
    }, {});

    if (info.name) {
        return {
            ...data,
            docsFolder: info.name
        };
    }
    const localedFolderDocName = getFolderDocName(lang);

    await Promise.resolve(
        api.request('documents.disk.createFolder', {
            folderPath: `${systemScansFolder}/${localedFolderDocName}`
        })
    );
    return {
        ...data,
        docsFolder: localedFolderDocName
    };
};

export const initNamedDocFolder = async (docId, scansFolder, docsFolder, lang) => {
    const docNames = Object.values(getDocNamesById(docId));
    const findDocRequests = docNames.map((docName) =>
        api.request('documents.disk.getResourceMeta', {
            folderPath: `${scansFolder}/${docsFolder}/${docName}`
        })
    );

    const results = await Promise.all(findDocRequests);

    const info = results.reduce((acc, res) => {
        if (res.path) {
            acc.name = res.name;
        }

        return acc;
    }, {});

    if (info.name) {
        return {
            docName: info.name
        };
    }
    const localedDocName = getLocalizedDocNameById(lang)(docId);

    return await Promise.resolve(
        api.request('documents.disk.createFolder', {
            folderPath: `${scansFolder}/${docsFolder}/${localedDocName}`
        })
    ).then(() => {
        return {
            docName: localedDocName
        };
    });
};

export const searchDocuments = async (docName) => {
    return await Promise.resolve(
        api.request('documents.disk.searchResource', {
            query: `документ:(${docName})`
        })
    ).then((searchMeta) => {
        return searchMeta.items ? searchMeta.items : [];
    });
};

export const getAllSearchDocuments = async (docName) => {
    const docs = [];

    let keepGoing = true;

    let offset = 0;
    const limit = 20;

    while (keepGoing) {
        const response = await Promise.resolve(
            api.request('documents.disk.searchResource', {
                query: `документ:(${docName})`,
                offset,
                limit,
                preview_size: 'XXL',
                preview_allow_big_size: true
            })
        ).then((searchMeta) => {
            return searchMeta.items ? searchMeta.items : [];
        });

        docs.push(...response);
        offset += limit;

        if (response.length < limit) {
            keepGoing = false;

            return docs;
        }
    }
};

export const getDocumentsInFolder = async (
    scansFolder,
    docsFolder,
    docName,
    options = {
        preview_size: 'XXL',
        preview_allow_big_size: true,
        limit: 20,
        offset: 0
    }
) => {
    return await Promise.resolve(
        api.request('documents.disk.getResourceMeta', {
            folderPath: `disk:/${scansFolder}/${docsFolder}/${docName}`,
            ...options
        })
    ).then((docInfo) => {
        return docInfo._embedded && docInfo._embedded.items ? docInfo._embedded.items : [];
    });
};

export const getAllDocumentsInFolder = async (scansFolder, docsFolder, docName) => {
    const docs = [];

    let keepGoing = true;

    let offset = 0;
    const limit = 20;

    while (keepGoing) {
        const response = await Promise.resolve(
            api.request('documents.disk.getResourceMeta', {
                folderPath: `disk:/${scansFolder}/${docsFolder}/${docName}`,
                preview_size: 'L',
                preview_allow_big_size: true,
                offset,
                limit
            })
        ).then((resourceMeta) => {
            return resourceMeta._embedded && resourceMeta._embedded.items ? resourceMeta._embedded.items : [];
        });

        docs.push(...response);
        offset += limit;

        if (response.length < limit) {
            keepGoing = false;

            return docs;
        }
    }
};

export const deleteDocument = async (path) => {
    return await Promise.resolve(
        api.request('documents.disk.deleteResource', {
            path
        })
    ).then((result) => result);
};

export const uploadDocument = async (path) => {
    return await Promise.resolve(api.request('documents.disk.createResource', {path})).then((result) => result);
};

export const getAllDocumentsByType = async (userId, documentType) => {
    const docs = [];

    let keepGoing = true;

    let offset = 0;
    const limit = 20;

    while (keepGoing) {
        const response = await Promise.resolve(
            api.request('documents.api.getDocumentList', {
                type: documentType,
                userId,
                options: {
                    offset,
                    limit
                }
            })
        ).then((result) => {
            return result;
        });

        if (response && response.documents && response.documents.length > 0) {
            docs.push(...response.documents);
        }
        offset += limit;

        if (!response.more) {
            keepGoing = false;

            return docs;
        }
    }
};

export const uploadDocumentByType = async (userId, type, file) => {
    const form = new FormData();

    form.append('document-uploader', file);
    form.append('userId', userId);
    form.append('type', type);

    return await Promise.resolve(
        api.request('documents.api.uploadDocument', form, {cache: false, processData: false, contentType: false})
    );
};

export const deleteDocumentById = async (userId, docId) => {
    return await Promise.resolve(api.request('documents.api.deleteDocument', {userId, docId})).then((result) => result);
};

export const copyDocument = async (userId, path, type) => {
    return await Promise.resolve(api.request('documents.api.copyDocument', {userId, path, type})).then(
        (result) => result
    );
};
