import api from '@blocks/api';
import utils from '@blocks/utils';
import {saveActionForRepeat, setEditMode} from '@blocks/common/actions';
import passport from '@plibs/pclientjs/js/passport';

export const GET_EMAILS = 'GET_EMAILS';
export const GET_EMAILS_SUCCESS = 'GET_EMAILS_SUCCESS';
export const GET_EMAILS_ERROR = 'GET_EMAILS_ERROR';
export const DELETE_EMAIL = 'DELETE_EMAIL';
export const DELETE_EMAIL_SUCCESS = 'DELETE_EMAIL_SUCCESS';
export const DELETE_EMAIL_ERROR = 'DELETE_EMAIL_ERROR';
export const SEND_CONFIRMATION_EMAIL = 'SEND_CONFIRMATION_EMAIL';
export const SEND_CONFIRMATION_EMAIL_SUCCESS = 'SEND_CONFIRMATION_EMAIL_SUCCESS';
export const SEND_CONFIRMATION_EMAIL_ERROR = 'SEND_CONFIRMATION_EMAIL_ERROR';
export const RESEND_CONFIRMATION_EMAIL = 'RESEND_CONFIRMATION_EMAIL';
export const RESEND_CONFIRMATION_EMAIL_SUCCESS = 'RESEND_CONFIRMATION_EMAIL_SUCCESS';
export const RESEND_CONFIRMATION_EMAIL_ERROR = 'RESEND_CONFIRMATION_EMAIL_ERROR';
export const CONFIRM_EMAIL_BY_CODE = 'CONFIRM_EMAIL_BY_CODE';
export const CONFIRM_EMAIL_BY_CODE_SUCCESS = 'CONFIRM_EMAIL_BY_CODE_SUCCESS';
export const CONFIRM_EMAIL_BY_CODE_ERROR = 'CONFIRM_EMAIL_BY_CODE_ERROR';
export const EMAIL_SET_SAFE = 'EMAIL_SET_SAFE';
export const EMAIL_SET_SAFE_SUCCESS = 'EMAIL_SET_SAFE_SUCCESS';
export const EMAIL_SET_SAFE_ERROR = 'EMAIL_SET_SAFE_ERROR';
export const CHANGE_EMAILS_STATE = 'CHANGE_EMAILS_STATE';
export const SET_EMAIL_ERROR = 'SET_EMAIL_ERROR';
export const TOGGLE_ALIASES_LIST = 'TOGGLE_ALIASES_LIST';
export const TOGGLE_EMAIL_SELECTION = 'TOGGLE_EMAIL_SELECTION';
export const SHOW_EMAIL_DELETE_REQUEST_POPUP = 'SHOW_EMAIL_DELETE_REQUEST_POPUP';
export const EMAILS_GOAL_PREFIX = 'profile_page_emails';

export const EMAILS_STATES = {
    root: 'root',
    list: 'list',
    add: 'add',
    sent: 'sent',
    confirmed: 'confirmed'
};

export const ERROR_MESSAGES = {
    'email.empty': i18n('_AUTH_.errors.required'),
    'email.invalid': i18n('_AUTH_.emails.errors.email-invalid'),
    'email.already_confirmed': i18n('_AUTH_.emails.errors.already-confirmed'),
    'email.already_sent': i18n('_AUTH_.emails.errors.already-sent'),
    'email.is_native': i18n('_AUTH_.emails.errors.is-native'),
    'email.incorrect_key': i18n('Profile.emails.errors.code-invalid'),
    'email.key_check_limit_exceeded': i18n('Profile.emails.errors.key-limit-exceeded'),
    'email.empty_key': i18n('_AUTH_.errors.required'),
    'account.not_found': i18n('Profile.emails.errors.not-found'),
    'account.disabled': i18n('Profile.emails.errors.not-found'),
    'account.disabled_on_deletion': i18n('Profile.emails.errors.not-found'),
    'sessionid.no_uid': i18n('Profile.emails.errors.not-found'),
    'email.other': i18n('Profile.addresses.errors.internal')
};

export const EMAILS_URLS = {
    list: '/profile/emails/list',
    'list-initial': '/profile/emails/list',
    add: '/profile/emails/add',
    'add-initial': '/profile/emails/add'
};

export function toggleAliasesList() {
    return {
        type: TOGGLE_ALIASES_LIST
    };
}

export function toggleEmailSelection(email) {
    return {
        type: TOGGLE_EMAIL_SELECTION,
        email
    };
}

export function showDeleteRequestPopup(email) {
    return {
        type: SHOW_EMAIL_DELETE_REQUEST_POPUP,
        email
    };
}

export function changeEmailsState(emailState) {
    return function(dispatch, getState) {
        const state = getState();
        const firstEmailState = state.emails.states[0];

        if (firstEmailState === EMAILS_STATES.add && emailState === EMAILS_STATES.root) {
            dispatch(setEditMode(null));
        } else {
            dispatch({
                type: CHANGE_EMAILS_STATE,
                state: emailState
            });
        }
    };
}

export function changeEmail(email) {
    return function(dispatch, getState) {
        const csrf = getState().common.csrf;
        const data = {email, csrf_token: csrf};

        dispatch({
            type: DELETE_EMAIL
        });
        dispatch(saveActionForRepeat(deleteEmail, email));
        api.request('email/delete', data)
            .done(function() {
                dispatch({
                    type: DELETE_EMAIL_SUCCESS,
                    email
                });
                dispatch({
                    type: CHANGE_EMAILS_STATE,
                    state: EMAILS_STATES.root
                });
            })
            .fail(function(err) {
                dispatch({
                    type: DELETE_EMAIL_ERROR,
                    errors: err.errors
                });
            });
    };
}

export function setError(error) {
    return {
        type: SET_EMAIL_ERROR,
        error
    };
}

export function getEmails() {
    return function(dispatch, getState) {
        const csrf = getState().common.csrf;

        dispatch({
            type: GET_EMAILS
        });
        api.request('email/list', {csrf_token: csrf})
            .done((res) => {
                dispatch({
                    type: GET_EMAILS_SUCCESS,
                    emails: res.emails
                });
            })
            .fail((err) => {
                dispatch({
                    type: GET_EMAILS_ERROR,
                    errors: err.errors
                });
            });
    };
}

export function addEmail(email) {
    return function(dispatch, getState) {
        const state = getState();
        const csrf = state.common.csrf;
        const havePassword = state.person.havePassword;
        const language = state.settings.language;

        let retpath = state.common.retpath;

        if (retpath) {
            if (!utils.isThirdPartyUrl(retpath)) {
                retpath = `https://${location.hostname}${retpath}`;
            }
        } else {
            retpath = `https://${location.hostname}/profile/${location.search}`;
        }
        const data = {
            is_safe: havePassword,
            csrf_token: csrf,
            email,
            retpath,
            validator_ui_url: `https://${location.hostname}${passport.basePath}email/confirm-by-link`,
            language
        };

        dispatch({
            type: SEND_CONFIRMATION_EMAIL
        });
        dispatch(saveActionForRepeat(addEmail, email));
        api.request('email/send-confirmation-email', data)
            .done(function(res) {
                dispatch({
                    type: SEND_CONFIRMATION_EMAIL_SUCCESS,
                    emails: res.emails,
                    email
                });
            })
            .fail(function(err) {
                dispatch({
                    type: SEND_CONFIRMATION_EMAIL_ERROR,
                    errors: err.errors
                });
            });
    };
}

export function deleteEmail(email) {
    return function(dispatch, getState) {
        const csrf = getState().common.csrf;
        const data = {email, csrf_token: csrf};

        dispatch({
            type: DELETE_EMAIL
        });
        dispatch(saveActionForRepeat(deleteEmail, email));
        api.request('email/delete', data)
            .done(function() {
                dispatch({
                    type: DELETE_EMAIL_SUCCESS,
                    email
                });
            })
            .fail(function(err) {
                if (Array.isArray(err.errors) && err.errors.indexOf('password.required') !== -1) {
                    dispatch(showDeleteRequestPopup(null));
                }

                dispatch({
                    type: DELETE_EMAIL_ERROR,
                    email,
                    errors: err.errors
                });
            });
    };
}

export function setSafeEmail(email) {
    return function(dispatch, getState) {
        const csrf = getState().common.csrf;
        const data = {
            csrf_token: csrf,
            email,
            is_safe: true
        };

        dispatch({
            type: EMAIL_SET_SAFE
        });
        dispatch(saveActionForRepeat(setSafeEmail, email));
        api.request('email/setup-confirmed', data)
            .done(function(res) {
                dispatch({
                    type: EMAIL_SET_SAFE_SUCCESS,
                    emails: res.emails
                });
            })
            .fail(function(err) {
                dispatch({
                    type: EMAIL_SET_SAFE_ERROR,
                    errors: err.errors
                });
            });
    };
}

export function confirmEmail(code) {
    return function(dispatch, getState) {
        const csrf = getState().common.csrf;
        const data = {
            csrf_token: csrf,
            key: code
        };

        dispatch({
            type: CONFIRM_EMAIL_BY_CODE
        });
        dispatch(saveActionForRepeat(confirmEmail, code));
        api.request('email/confirm-by-code', data)
            .done(function(res) {
                dispatch({
                    type: CONFIRM_EMAIL_BY_CODE_SUCCESS,
                    emails: res.emails
                });
            })
            .fail(function(err) {
                dispatch({
                    type: CONFIRM_EMAIL_BY_CODE_ERROR,
                    errors: err.errors
                });
            });
    };
}

export function resendCode(email) {
    return function(dispatch, getState) {
        const state = getState();
        const csrf = state.common.csrf;
        const havePassword = state.person.havePassword;
        const language = state.settings.language;

        let retpath = state.common.retpath;

        if (retpath) {
            if (!(retpath.slice(0, 5) === 'https')) {
                retpath = `https://${location.hostname}${retpath}`;
            }
        } else {
            retpath = `https://${location.hostname}/profile/${location.search}`;
        }
        const data = {
            email,
            is_safe: havePassword,
            csrf_token: csrf,
            retpath,
            // eslint-disable-next-line compat/compat
            validator_ui_url: `${location.origin}${passport.basePath}email/confirm-by-link`,
            language
        };

        dispatch({
            type: RESEND_CONFIRMATION_EMAIL
        });
        dispatch(saveActionForRepeat(resendCode, email));
        api.request('email/send-confirmation-email', data)
            .done(function() {
                dispatch({
                    type: RESEND_CONFIRMATION_EMAIL_SUCCESS
                });
            })
            .fail(function(err) {
                dispatch({
                    type: RESEND_CONFIRMATION_EMAIL_ERROR,
                    errors: err.errors
                });
            });
    };
}
