import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {push} from 'react-router-redux';
import classnames from 'classnames';
import escape from 'lodash/escape';
import {Input} from '@components/Input';
import {Spin} from '@components/Spin';
import {Link} from '@components/Link';
import {Button} from '@components/Button';
import ReturnToService from '@blocks/morda/components/return_button/return_button.jsx';
import {
    addEmail,
    changeEmail,
    confirmEmail,
    resendCode,
    setError,
    changeEmailsState,
    EMAILS_GOAL_PREFIX,
    EMAILS_STATES,
    EMAILS_URLS,
    ERROR_MESSAGES
} from '../actions';
import {MAIN_PAGE_URL} from '@blocks/constants';
import metrics from '@blocks/metrics.js';

const DESCRIPTIONS = {
    sent: i18n('Profile.emails.page.description.confirm'),
    confirmed: i18n('Profile.emails.page.description.confirmed')
};

function getDescription(type, email) {
    const d = DESCRIPTIONS[type];

    return d.replace('%1', `<strong style='color: black;'>${escape(email)}</strong>`);
}

class AddEmail extends Component {
    constructor(props, context) {
        super(props, context);

        this.closeModal = this.closeModal.bind(this);
        this.handleSubmit = this.handleSubmit.bind(this);
        this.changeEmail = this.changeEmail.bind(this);
        this.resendCode = this.resendCode.bind(this);
        this.onFieldChange = this.onFieldChange.bind(this);

        this.state = {
            email: '',
            code: ''
        };
    }

    closeModal() {
        metrics.goal(`${EMAILS_GOAL_PREFIX}_close`);

        this.setState({
            email: '',
            code: ''
        });

        this.props.dispatch(setError(null));
        this.props.dispatch(changeEmailsState(EMAILS_STATES.root));
    }

    onFieldChange(event) {
        const {target = {}} = event;
        const {value, name} = target;

        this.setState({[name]: value});
    }

    changeEmail() {
        this.props.dispatch(changeEmail(this.props.addedEmail));
    }

    resendCode() {
        this.props.dispatch(resendCode(this.props.addedEmail));
    }

    handleSubmit(event) {
        const {dispatch, addEmailState} = this.props;

        event.preventDefault();
        dispatch(setError(null));
        switch (addEmailState) {
            case EMAILS_STATES.add: {
                const email = this.state.email;

                if (!email) {
                    dispatch(setError('email.empty'));
                } else {
                    this.setState({email: ''});
                    metrics.goal(`${EMAILS_GOAL_PREFIX}_add_email`);
                    dispatch(addEmail(email));
                }
                break;
            }
            case EMAILS_STATES.sent: {
                const code = (this.state.code || '').trim();

                if (!code) {
                    dispatch(setError('email.empty_key'));
                } else {
                    metrics.goal(`${EMAILS_GOAL_PREFIX}_confirm_email_code`);
                    dispatch(confirmEmail(code.trim()));
                }
                this.setState({code: ''});
                break;
            }
            case EMAILS_STATES.confirmed: {
                const {retpath = MAIN_PAGE_URL, modal} = this.props;

                metrics.goal(`${EMAILS_GOAL_PREFIX}_email_confirmed`);
                if (modal) {
                    dispatch(changeEmailsState(EMAILS_STATES.root));
                } else if (retpath) {
                    location.href = retpath;
                } else {
                    dispatch(push('/profile'));
                }
                break;
            }
            default:
                break;
        }
    }

    componentWillReceiveProps(nextProps) {
        if (nextProps.addEmailState === EMAILS_STATES.list && !this.props.modal) {
            this.props.dispatch(push(EMAILS_URLS.list));
        }
    }

    render() {
        const {addEmailState, isMobile, loading, retpath, addedEmail, error, modal, person = {}} = this.props;
        const {havePassword} = person;

        return (
            <form
                className={classnames('p-control-form', `emails-add_${addEmailState}`, {
                    'p-control-modal': modal
                })}
                onSubmit={this.handleSubmit}
            >
                <h2 className='section-title'>
                    {havePassword ? i18n('_AUTH_.emails.page.title') : i18n('Profile.emails.notify.title')}
                </h2>

                {(() => {
                    if (addEmailState === EMAILS_STATES.add) {
                        return (
                            <div>
                                <div className='section-info'>
                                    {havePassword
                                        ? i18n('Profile.emails.page.description.add-email')
                                        : i18n('Profile.emails.page.description.add-notify-email')}
                                </div>
                                <div className='emails-add__field'>
                                    <label className='emails-add__input-label'>
                                        {i18n('Profile.emails.page.label.add-email')}:
                                    </label>
                                    <div className='emails-add__input-field'>
                                        <Input
                                            value={this.state.email}
                                            name='email'
                                            onChange={this.onFieldChange}
                                            size='l'
                                        />
                                    </div>
                                    {Boolean(error) && (
                                        <div className='p-control-error-block'>{ERROR_MESSAGES[error]}</div>
                                    )}
                                </div>
                                <div className='clearfix'>
                                    {modal ? (
                                        <div className='p-control-saveblock-cell-left p-control-saveblock-button'>
                                            <Button
                                                view='pseudo'
                                                size='l'
                                                type='button'
                                                text={i18n('_AUTH_.common.cancel')}
                                                onClick={this.closeModal}
                                                width={isMobile ? 'max' : 'auto'}
                                            />
                                        </div>
                                    ) : (
                                        <div className='p-control-saveblock-cell-left p-control-saveblock-button'>
                                            <ReturnToService retpath={retpath} isModal={false} isMobile={isMobile} />
                                        </div>
                                    )}
                                    <div className='p-control-saveblock-cell-right p-control-saveblock-button'>
                                        <Button
                                            view='action'
                                            type='submit'
                                            size='l'
                                            text={i18n('Profile.common.add')}
                                            width={isMobile ? 'max' : 'auto'}
                                        />
                                    </div>
                                </div>
                            </div>
                        );
                    }

                    if (addEmailState === EMAILS_STATES.sent) {
                        return (
                            <div>
                                <div
                                    className='emails-add__description'
                                    dangerouslySetInnerHTML={{
                                        __html: getDescription('sent', addedEmail)
                                    }}
                                />
                                <div className='emails-add__field clearfix'>
                                    <label className='emails-add__input-label'>
                                        {i18n('Profile.emails.page.label.code')}:
                                    </label>
                                    <div className='emails-add__input-field'>
                                        <Input
                                            value={this.state.code}
                                            name='code'
                                            onChange={this.onFieldChange}
                                            size='l'
                                        />
                                        {Boolean(error) && (
                                            <div className='p-control-error-block'>{ERROR_MESSAGES[error]}</div>
                                        )}
                                    </div>
                                    <div className='emails-add__controls'>
                                        {loading === 'email' || loading === 'deleteEmail' ? (
                                            <Spin progress={true} size='s' />
                                        ) : (
                                            <div>
                                                <div>
                                                    <Link
                                                        className='change-email-link'
                                                        pseudo={true}
                                                        onClick={this.changeEmail}
                                                    >
                                                        {i18n('Profile.emails.change-email')}
                                                    </Link>
                                                </div>
                                                <div>
                                                    <Link
                                                        className='resend-code-link'
                                                        pseudo={true}
                                                        onClick={this.resendCode}
                                                    >
                                                        {i18n('Profile.emails.page.resend-code')}
                                                    </Link>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </div>
                                <div className='clearfix'>
                                    {Boolean(modal) && (
                                        <div className='p-control-saveblock-cell-left p-control-saveblock-button'>
                                            <Button
                                                view='pseudo'
                                                size='l'
                                                text={i18n('_AUTH_.common.cancel')}
                                                onClick={this.closeModal}
                                                width={isMobile ? 'max' : 'auto'}
                                            />
                                        </div>
                                    )}
                                    <div className='p-control-saveblock-cell-right p-control-saveblock-button'>
                                        <Button
                                            type='submit'
                                            size='l'
                                            view='action'
                                            text={i18n('Profile.common.confirm')}
                                            width={isMobile ? 'max' : 'auto'}
                                        />
                                    </div>
                                </div>
                            </div>
                        );
                    }

                    if (addEmailState === 'confirmed') {
                        return (
                            <div>
                                <div className='emails-add__description'>
                                    <div style={{display: 'tableRow'}}>
                                        <div className='emails-icon__success' />
                                        <div
                                            className='emails-add__description-text'
                                            dangerouslySetInnerHTML={{
                                                __html: getDescription('confirmed', addedEmail)
                                            }}
                                        />
                                    </div>
                                </div>
                                <div className='clearfix'>
                                    <div className='p-control-saveblock-cell-right p-control-saveblock-button'>
                                        <Button
                                            type='submit'
                                            size='l'
                                            view='action'
                                            text={i18n('Profile.common.ok')}
                                            width={isMobile ? 'max' : 'auto'}
                                        />
                                    </div>
                                </div>
                            </div>
                        );
                    }

                    return null;
                })()}
            </form>
        );
    }
}

AddEmail.propTypes = {
    dispatch: PropTypes.func.isRequired,
    addedEmail: PropTypes.string,
    addEmailState: PropTypes.string,
    retpath: PropTypes.string,
    isMobile: PropTypes.bool.isRequired,
    error: PropTypes.string,
    loading: PropTypes.string,
    modal: PropTypes.bool.isRequired,
    person: PropTypes.shape({
        havePassword: PropTypes.bool
    })
};

export default AddEmail;
