import React, {Component} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {push} from 'react-router-redux';
import {Input} from '@components/Input';
import {Spin} from '@components/Spin';
import {Link as LegoLink} from '@components/Link';
import {Button} from '@components/Button';
import {Popup} from '@components/Popup';
import Link from '@blocks/morda/components/link/link.jsx';
import MoreEmailsLink from '@blocks/morda/emails/more_emails_link.jsx';
import ReturnToService from '@blocks/morda/components/return_button/return_button.jsx';
import Collapse from '@blocks/morda/emails/collapse.jsx';
import metrics from '@blocks/metrics.js';
import {
    setError,
    addEmail,
    deleteEmail,
    setSafeEmail,
    toggleAliasesList,
    toggleEmailSelection,
    showDeleteRequestPopup,
    EMAILS_GOAL_PREFIX,
    EMAILS_STATES,
    EMAILS_URLS,
    ERROR_MESSAGES
} from '../actions';
import {VALIDATOR_SUPPORT_URL, SUPPORT_DOMAINS} from '../constants';

class EmailsList extends Component {
    constructor(props, context) {
        super(props, context);
        const domain = SUPPORT_DOMAINS.find((el) => el === props.tld) ? props.tld : 'ru';

        this.validatorSupportURL = VALIDATOR_SUPPORT_URL.replace('%tld%', domain);
        this.addEmail = this.addEmail.bind(this);
        this.deleteEmail = this.deleteEmail.bind(this);
        this.confirmEmail = this.confirmEmail.bind(this);
        this.setEmailForRestore = this.setEmailForRestore.bind(this);
        this.toggleEmailSelection = this.toggleEmailSelection.bind(this);
        this.toggleAliasesList = this.toggleAliasesList.bind(this);
        this.showDeleteRequestPopup = this.showDeleteRequestPopup.bind(this);
        this.hideDeleteRequestPopup = this.hideDeleteRequestPopup.bind(this);
        this.onFieldChange = this.onFieldChange.bind(this);

        this.state = {
            email: ''
        };
    }

    componentWillUnmount() {
        const {dispatch, openAliasesList} = this.props;

        if (openAliasesList) {
            dispatch(toggleAliasesList());
        }
    }

    addEmail(event) {
        const email = this.state.email;

        event.preventDefault();
        if (!email) {
            this.props.dispatch(setError('email.empty'));
        } else {
            this.setState({email: ''});
            this.props.dispatch(setError(null));
            metrics.goal(`${EMAILS_GOAL_PREFIX}_list_add_email`);
            this.props.dispatch(addEmail(email));
        }
    }

    confirmEmail(event, email) {
        event.preventDefault();
        this.setState({email: ''});
        this.props.dispatch(setError(null));
        metrics.goal(`${EMAILS_GOAL_PREFIX}_confirm_added_email`);
        this.props.dispatch(addEmail(email));
    }

    onFieldChange(event) {
        const {target = {}} = event;
        const {value} = target;

        this.setState({email: value});
    }

    setEmailForRestore(event, email) {
        this.props.dispatch(setSafeEmail(email));
    }

    deleteEmail(event, deletedEmail, section) {
        let email;

        if (section && section === 'other') {
            email = deletedEmail;
        } else {
            email = this.props.deleteRequestEmail;
        }

        this.props.dispatch(deleteEmail(email));
    }

    toggleEmailSelection(event, email) {
        const dispatch = this.props.dispatch;

        if (!this.props.isMobile) {
            return;
        }
        dispatch(toggleEmailSelection(email));
    }

    toggleAliasesList(event) {
        event.preventDefault();
        this.props.dispatch(toggleAliasesList());
    }

    showDeleteRequestPopup(event, email) {
        this.props.dispatch(showDeleteRequestPopup(email));
    }

    hideDeleteRequestPopup() {
        this.props.dispatch(showDeleteRequestPopup(null));
    }

    componentWillReceiveProps(nextProps) {
        if (nextProps.lastState === EMAILS_STATES.sent && !this.props.modal) {
            this.props.dispatch(push(EMAILS_URLS.add));
        }
    }

    render() {
        const {
            loading,
            haveEmails,
            error,
            isMobile,
            retpath,
            selectedEmail,
            deleteRequestEmail,
            modal,
            openAliasesList,
            isLite,
            emails: {forRestore, native, forNotifications, rpop, other},
            person = {}
        } = this.props;
        const {isLiteUser, havePassword, login} = person;
        const canRemove = (email) => (!isLiteUser && havePassword) || (isLiteUser && login !== email.name);

        return (
            <form className={classnames('p-control-form')} onSubmit={this.addEmail} data-t='emails-list'>
                <h2 className='section-title'>{i18n('Profile.emails.page.title.list')}</h2>
                {!isLite && (
                    <div className='section-info'>
                        {i18n('Profile.emails.page.description.list')}{' '}
                        <a href={this.validatorSupportURL} target='_blank' className='b-link' rel='noopener noreferrer'>
                            {i18n('Profile.common.read-more')}
                        </a>
                        <span>.</span>
                    </div>
                )}
                <div className='emails-list__add-email clearfix'>
                    <label className='emails-add__input-label'>
                        {havePassword
                            ? i18n('Profile.emails.page.label.list')
                            : i18n('Profile.emails.notify.label.list')}
                        :
                    </label>
                    <div className='emails-list__input'>
                        <Input
                            value={this.state.email}
                            placeholder={i18n('Profile.emails.page.placeholder.list')}
                            onChange={this.onFieldChange}
                            size='l'
                        />
                        {Boolean(error) && <div className='p-control-error-block'>{ERROR_MESSAGES[error]}</div>}
                    </div>
                    <div className='emails-list__add-email-button'>
                        <Button size='l' text={i18n('Profile.emails.page.add-address')} view='action' type='submit' />
                    </div>
                </div>

                {loading === 'widget' && (
                    <div className='emails-list__spinner'>
                        <Spin progress={true} size='s' />
                    </div>
                )}

                {haveEmails && (
                    <div className='emails-list__list'>
                        {Boolean(native.length) && (
                            <div className='emails-list__list-box' data-t='section:native'>
                                <div className='emails-list__list-header'>
                                    {i18n('Profile.emails.aliases')}
                                    <div className='emails-list__list-icon social-widget__icon' />
                                </div>
                                <div className='emails-list__list-label'>
                                    {i18n('Profile.emails.aliases.description')}
                                </div>
                                {Boolean(native[0].name) && (
                                    <div className='emails-list__list-item'>
                                        <div className='emails-list__email emails-list__email_confirmed'>
                                            {native[0].name}
                                        </div>
                                    </div>
                                )}
                                <Collapse isOpen={openAliasesList}>
                                    <div>
                                        {native.slice(1).map((email) => (
                                            <div key={email.name} className='emails-list__list-item'>
                                                <div className='emails-list__email emails-list__email_confirmed'>
                                                    {email.name}
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </Collapse>
                                <div className='emails-list__show-more'>
                                    {openAliasesList ? (
                                        <Link onClick={this.toggleAliasesList} url={''}>
                                            {i18n('Profile.common.show-less')}
                                        </Link>
                                    ) : (
                                        <MoreEmailsLink
                                            url={''}
                                            limit={1}
                                            onClick={this.toggleAliasesList}
                                            count={native.length}
                                        />
                                    )}
                                </div>
                            </div>
                        )}

                        {Boolean(forRestore.length > 0) && (
                            <div className='emails-list__list-box' data-t='section:restore'>
                                <div className='emails-list__list-header'>
                                    {i18n('Profile.emails.page.for-restore')}
                                    <div className='emails-list__list-icon emails-icon__for-restore' />
                                </div>
                                <div className='emails-list__list-label'>
                                    {i18n('Profile.emails.page.for-restore.description')}
                                </div>
                                {forRestore.map((email) => {
                                    const emailRef = React.createRef();

                                    return (
                                        <div
                                            className={classnames('emails-list__list-item clearfix', {
                                                'emails-list__list-item_selected': email.name === selectedEmail
                                            })}
                                            ref={emailRef}
                                            key={email.name}
                                            onClick={(event) => this.toggleEmailSelection(event, email.name)}
                                        >
                                            <div
                                                className={classnames('emails-list__email', {
                                                    'emails-list__email_confirmed': email.confirmed
                                                })}
                                            >
                                                {email.name}
                                            </div>
                                            {!isMobile && canRemove(email) && (
                                                <div
                                                    className='emails-list__delete-email emails-icon__delete-email'
                                                    onClick={(event) => this.showDeleteRequestPopup(event, email.name)}
                                                />
                                            )}

                                            {!email.confirmed && !isMobile && (
                                                <span>
                                                    <LegoLink
                                                        className='emails-list__confirm-email'
                                                        pseudo={true}
                                                        onClick={(event) => this.confirmEmail(event, email.name)}
                                                    >
                                                        {i18n('Profile.common.confirm')}
                                                    </LegoLink>
                                                    <div className='emails-list__email-not-confirmed'>
                                                        {i18n('Profile.emails.page.not-confirmed')}
                                                    </div>
                                                </span>
                                            )}

                                            {Boolean(isMobile && email.name === selectedEmail) && (
                                                <div>
                                                    {!email.confirmed && (
                                                        <div className='emails-list__confirm-button'>
                                                            <Button
                                                                view='white'
                                                                size='l'
                                                                width='max'
                                                                onClick={(event) =>
                                                                    this.confirmEmail(event, email.name)
                                                                }
                                                            >
                                                                {i18n('Profile.common.confirm')}
                                                            </Button>
                                                        </div>
                                                    )}

                                                    {canRemove(email) && (
                                                        <div className='emails-list__delete-button'>
                                                            <Button
                                                                view='pseudo'
                                                                size='l'
                                                                width='max'
                                                                onClick={(event) =>
                                                                    this.showDeleteRequestPopup(event, email.name)
                                                                }
                                                            >
                                                                {i18n('Profile.common.delete')}
                                                            </Button>
                                                        </div>
                                                    )}
                                                </div>
                                            )}

                                            <Popup
                                                zIndex={99999}
                                                hasTail={true}
                                                directions={['right-center', 'top-right', 'top-center']}
                                                target='anchor'
                                                anchor={emailRef}
                                                onOutsideClick={this.hideDeleteRequestPopup}
                                                ignoreRefs={[emailRef]}
                                                visible={deleteRequestEmail === email.name}
                                            >
                                                <div className='emails-list__popup'>
                                                    <div className='emails-list__popup-text'>
                                                        {i18n('Profile.emails.popup.delete-confirmed')}
                                                    </div>
                                                    <div className='emails-list__popup-buttons'>
                                                        <Button
                                                            text={i18n('Profile.common.delete')}
                                                            size='l'
                                                            view='action'
                                                            type='button'
                                                            onClick={(event) => this.deleteEmail(event, email.name)}
                                                            dataT='common-email-delete'
                                                        />
                                                        <Button
                                                            text={i18n('_AUTH_.common.cancel')}
                                                            size='l'
                                                            view='pseudo'
                                                            type='button'
                                                            onClick={this.hideDeleteRequestPopup}
                                                            dataT='cancel-email-delete'
                                                        />
                                                    </div>
                                                </div>
                                            </Popup>
                                        </div>
                                    );
                                })}
                            </div>
                        )}

                        {Boolean(forNotifications.length > 0) && (
                            <div className='emails-list__list-box' data-t='section:notify'>
                                <div className='emails-list__list-header'>
                                    {i18n('Profile.emails.page.for-notify')}
                                    <div className='emails-list__list-icon emails-icon__for-notify' />
                                </div>
                                <div className='emails-list__list-label'>
                                    {havePassword
                                        ? i18n('Profile.emails.page.for-notify.description')
                                        : i18n('Profile.emails.page.for-notify.description.soc')}
                                </div>

                                {forNotifications.map((email) => {
                                    const emailRef = React.createRef();

                                    return (
                                        <div key={email.name}>
                                            <div
                                                className={classnames('emails-list__list-item clearfix', {
                                                    'emails-list__list-item_selected': email.name === selectedEmail
                                                })}
                                                onClick={(event) => this.toggleEmailSelection(event, email.name)}
                                                ref={emailRef}
                                            >
                                                <div
                                                    className={classnames('emails-list__email', {
                                                        'emails-list__email_confirmed': email.confirmed
                                                    })}
                                                >
                                                    {email.name}
                                                </div>

                                                {!isMobile && (
                                                    <div
                                                        className='emails-list__delete-email emails-icon__delete-email'
                                                        onClick={(event) =>
                                                            this.showDeleteRequestPopup(event, email.name)
                                                        }
                                                    />
                                                )}

                                                {Boolean(havePassword && !isMobile && email.confirmed) && (
                                                    <LegoLink
                                                        className={`emails-list__confirm-email
                                                        emails-list__confirm-email_for-notify`}
                                                        pseudo={true}
                                                        onClick={(event) => this.setEmailForRestore(event, email.name)}
                                                    >
                                                        {i18n('Profile.emails.page.use-for-restore')}
                                                    </LegoLink>
                                                )}

                                                {!email.confirmed && !isMobile && (
                                                    <span>
                                                        <LegoLink
                                                            className='emails-list__confirm-email'
                                                            pseudo={true}
                                                            onClick={(event) => this.confirmEmail(event, email.name)}
                                                        >
                                                            {i18n('Profile.common.confirm')}
                                                        </LegoLink>
                                                        <div className='emails-list__email-not-confirmed'>
                                                            {i18n('Profile.emails.page.not-confirmed')}
                                                        </div>
                                                    </span>
                                                )}

                                                {Boolean(isMobile && email.name === selectedEmail) && (
                                                    <div>
                                                        {Boolean(havePassword && email.confirmed) && (
                                                            <div
                                                                className='emails-list__confirm-button'
                                                                onClick={(event) =>
                                                                    this.setEmailForRestore(event, email.name)
                                                                }
                                                            >
                                                                {i18n('Profile.emails.page.use-for-restore')}
                                                            </div>
                                                        )}
                                                        {!email.confirmed && (
                                                            <div
                                                                className='emails-list__confirm-button'
                                                                onClick={(event) =>
                                                                    this.confirmEmail(event, email.name)
                                                                }
                                                            >
                                                                {i18n('Profile.common.confirm')}
                                                            </div>
                                                        )}
                                                        <div
                                                            className='emails-list__delete-button'
                                                            onClick={(event) =>
                                                                this.showDeleteRequestPopup(event, email.name)
                                                            }
                                                        >
                                                            {i18n('Profile.common.delete')}
                                                        </div>
                                                    </div>
                                                )}
                                            </div>

                                            <Popup
                                                zIndex={99999}
                                                hasTail={true}
                                                target='anchor'
                                                directions={['right-center', 'top-right', 'top-center']}
                                                anchor={emailRef}
                                                onOutsideClick={this.hideDeleteRequestPopup}
                                                ignoreRefs={[emailRef]}
                                                visible={deleteRequestEmail === email.name}
                                            >
                                                <div className='emails-list__popup'>
                                                    <div className='emails-list__popup-text'>
                                                        {i18n('Profile.emails.popup.delete-notify')}
                                                    </div>
                                                    <div className='emails-list__popup-buttons'>
                                                        <Button
                                                            text={i18n('Profile.common.delete')}
                                                            size='l'
                                                            view='action'
                                                            type='button'
                                                            onClick={(event) => this.deleteEmail(event, email.name)}
                                                        />
                                                        <Button
                                                            text={i18n('_AUTH_.common.cancel')}
                                                            size='l'
                                                            view='pseudo'
                                                            type='button'
                                                            onClick={this.hideDeleteRequestPopup}
                                                        />
                                                    </div>
                                                </div>
                                            </Popup>
                                        </div>
                                    );
                                })}
                            </div>
                        )}

                        {Boolean(rpop.length > 0) && (
                            <div className='emails-list__list-box' data-t='section:rpop'>
                                <div className='emails-list__list-header'>
                                    {i18n('Profile.emails.page.for-collect')}
                                    <div className='emails-list__list-icon emails-icon__for-collect' />
                                </div>
                                <div className='emails-list__list-label'>
                                    {i18n('Profile.emails.page.for-collect.description')}
                                </div>
                                {rpop.map((email) => {
                                    const emailRef = React.createRef();

                                    return (
                                        <div key={email.name}>
                                            <div
                                                className={classnames('emails-list__list-item clearfix', {
                                                    'emails-list__list-item_selected': email.name === selectedEmail
                                                })}
                                                onClick={(event) => this.toggleEmailSelection(event, email.name)}
                                                ref={emailRef}
                                            >
                                                <div className='emails-list__email emails-list__email_confirmed'>
                                                    {email.name}
                                                </div>
                                                {Boolean(email.confirmed) && (
                                                    <React.Fragment>
                                                        {!isMobile && (
                                                            <div
                                                                // eslint-disable-next-line
                                                                className='emails-list__delete-email emails-icon__delete-email'
                                                                onClick={(event) =>
                                                                    this.showDeleteRequestPopup(event, email.name)
                                                                }
                                                            />
                                                        )}
                                                        {Boolean(isMobile && email.name === selectedEmail) && (
                                                            <div
                                                                className='emails-list__delete-button'
                                                                onClick={(event) =>
                                                                    this.showDeleteRequestPopup(event, email.name)
                                                                }
                                                            >
                                                                {i18n('Profile.common.delete')}
                                                            </div>
                                                        )}
                                                    </React.Fragment>
                                                )}
                                            </div>
                                            <Popup
                                                zIndex={99999}
                                                hasTail={true}
                                                target='anchor'
                                                directions={['right-center', 'top-right', 'top-center']}
                                                anchor={emailRef}
                                                onOutsideClick={this.hideDeleteRequestPopup}
                                                ignoreRefs={[emailRef]}
                                                visible={deleteRequestEmail === email.name}
                                            >
                                                <div className='emails-list__popup'>
                                                    <div className='emails-list__popup-text'>
                                                        {i18n('Profile.emails.popup.delete-rpop')}
                                                    </div>
                                                    <div className='emails-list__popup-buttons'>
                                                        <Button
                                                            text={i18n('Profile.common.delete')}
                                                            size='l'
                                                            view='action'
                                                            type='button'
                                                            onClick={(event) => this.deleteEmail(event, email.name)}
                                                        />
                                                        <Button
                                                            text={i18n('_AUTH_.common.cancel')}
                                                            size='l'
                                                            view='pseudo'
                                                            type='button'
                                                            onClick={this.hideDeleteRequestPopup}
                                                        />
                                                    </div>
                                                </div>
                                            </Popup>
                                        </div>
                                    );
                                })}
                            </div>
                        )}

                        {Boolean(other.length > 0) && (
                            <div className='emails-list__list-box' data-t='section:other'>
                                <div className='emails-list__list-header'>{i18n('Profile.emails.page.other')}</div>
                                {other.map((email) => (
                                    <div key={email.name} className='emails-list__list-item clearfix'>
                                        <div
                                            className='emails-list__email emails-list__email_confirmed'
                                            onClick={(event) => this.toggleEmailSelection(event, email.name)}
                                        >
                                            {email.name}
                                        </div>
                                        {!isMobile && (
                                            <div
                                                className='emails-list__delete-email emails-icon__delete-email'
                                                onClick={(event) => this.deleteEmail(event, email.name, 'other')}
                                            />
                                        )}
                                        {Boolean(isMobile && email.name === selectedEmail) && (
                                            <div
                                                className='emails-list__delete-button'
                                                onClick={(event) => this.deleteEmail(event, email.name, 'other')}
                                            >
                                                {i18n('Profile.common.delete')}
                                            </div>
                                        )}
                                    </div>
                                ))}
                            </div>
                        )}
                    </div>
                )}

                <ReturnToService
                    className='emails-list__return-button'
                    retpath={retpath}
                    isModal={modal}
                    isMobile={isMobile}
                />
            </form>
        );
    }
}

EmailsList.propTypes = {
    isLite: PropTypes.bool,
    dispatch: PropTypes.func.isRequired,
    tld: PropTypes.string.isRequired,
    selectedEmail: PropTypes.string,
    deleteRequestEmail: PropTypes.string,
    openAliasesList: PropTypes.bool.isRequired,
    isMobile: PropTypes.bool.isRequired,
    modal: PropTypes.bool.isRequired,
    loading: PropTypes.bool.isRequired,
    retpath: PropTypes.string,
    haveEmails: PropTypes.bool.isRequired,
    error: PropTypes.string,
    emails: PropTypes.shape({
        native: PropTypes.array.isRequired,
        forRestore: PropTypes.array.isRequired,
        forNotifications: PropTypes.array.isRequired,
        rpop: PropTypes.array.isRequired,
        other: PropTypes.array.isRequired
    }).isRequired,
    person: PropTypes.object,
    lastState: PropTypes.string
};

export default EmailsList;
