import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';

import {Link} from '@components/Link';
import Modal from '@components/Wrappers/Modal.jsx';

import {Section} from '@blocks/morda/new/section/section.jsx';
import AddressesControl from '@blocks/morda/addresses/components/addresses_control';

import Map from '@blocks/morda/components/map/map.jsx';
import Toggle from '@blocks/morda/new/toggle/toggle.jsx';

import withMethods from '@blocks/hocs/withMethods';

import metrics from '@blocks/metrics';

import {
    EDIT_MODE_ADDRESSES,
    EDIT_MODE_DELIVERY_ADDRESSES,
    ADDRESSES_LINK,
    ADDRESSES_GOAL_PREFIX,
    ADDRESS_TYPES
} from '@blocks/morda/addresses/actions';

import * as extracted from './addresses_block.js';
import {getAddresses} from '@blocks/morda/addresses/actions/get_addresses';

import {EditSvg} from './editSvg';
import {PlusSvg} from './plusSvg';

const b = cn('Addresses');

const MAX_DELIVERY_SHOWN = 1;

export class AddressesBlock extends React.Component {
    constructor(props) {
        super(props);

        this.showAddresses = extracted.showAddresses.bind(this);
        this.showDeliveryAddresses = extracted.showDeliveryAddresses.bind(this);
        this.getAddressLink = extracted.getAddressLink.bind(this);
        this.setEditMode = extracted.setMode.bind(this);
        this.isTouch = props.settings.isTouch;
    }

    componentDidMount() {
        const {dispatch} = this.props;

        dispatch(getAddresses());
    }

    getLinkText(text, isPlus) {
        const {common: {v2 = false} = {}} = this.props;

        if (!v2 || !text) {
            return text;
        }

        const SvgComponent = isPlus ? PlusSvg : EditSvg;

        const splitText = text.split(' ');
        const lastWord = splitText.pop();
        const v2Text = `${splitText.join(' ')} `;

        return (
            <div className={b('link')}>
                {v2Text}
                <span className={b('linkLast')}>
                    {lastWord}
                    <SvgComponent className={b('svg')} />
                </span>
            </div>
        );
    }

    render() {
        const {addresses = {}, settings = {}, methods, common = {}, dispatch} = this.props;
        const {v2 = false} = common;
        const {
            delivery = [],
            home = {
                id: 'home',
                editedAddressLine: ''
            },
            work = {
                id: 'work',
                editedAddressLine: ''
            },
            successConnect = false
        } = addresses;
        const TUNE_PLACES = (settings.tunePlaces || '').replace('%tld%', settings.tld);
        const MARKET_ADDRESSES = (settings.marketAddresses || '').replace('%tld%', settings.tld);
        const mapAddrs = [];
        const linkParams = {
            pseudo: false,
            onClick: this.showAddresses
        };
        const deLinkParams = {
            pseudo: false,
            onClick: this.showDeliveryAddresses
        };
        const items = delivery.map((item) =>
            this.getAddressLink(
                this.getLinkText(item.addressLineShort || item.addressLine, false),
                this.showDeliveryAddresses
            )
        );
        const count = delivery.length - MAX_DELIVERY_SHOWN;
        const toggleText = i18n(
            {
                count,
                one: 'Profile.addresses.more.one',
                some: 'Profile.addresses.more.some',
                many: 'Profile.addresses.more.many',
                none: ''
            },
            count
        );

        if (this.isTouch && successConnect) {
            linkParams.url = deLinkParams.url = ADDRESSES_LINK;
        } else if (successConnect) {
            linkParams.pseudo = deLinkParams.pseudo = true;
            linkParams.url = deLinkParams.url = undefined;
        } else {
            linkParams.url = TUNE_PLACES;
            deLinkParams.url = MARKET_ADDRESSES;
        }

        if (home.longitude) {
            mapAddrs.push(home);
        }

        if (work.longitude) {
            mapAddrs.push(work);
        }

        return (
            <Section className={b({v2})} icon='maps' title={i18n('Profile.addresses.title.short')}>
                <div className={b('list')}>
                    {Boolean(home.longitude) || Boolean(work.longitude) ? (
                        [home, work].map((item) => {
                            const address = item.addressLineShort || item.addressLine;

                            return (
                                <div className={b('block')} key={item.id}>
                                    <div className='s-block__sub'>{ADDRESS_TYPES[item.id]}</div>
                                    <div className='s-block__sub s-block__sub_hl'>
                                        <span onClick={methods.editNavi} data-t={`pssp:edit-${item.id}`}>
                                            {this.getAddressLink(
                                                this.getLinkText(address || i18n('Profile.addresses.add'), !address),
                                                this.showAddresses
                                            )}
                                        </span>
                                    </div>
                                </div>
                            );
                        })
                    ) : (
                        <div className={b('block')}>
                            <div className='s-block__sub s-block__sub_hl'>
                                <span onClick={methods.addNavi} data-t='pssp:add-home-work'>
                                    <Link href={linkParams.url} onClick={linkParams.onClick}>
                                        {this.getLinkText(i18n('Profile.addresses.add.defaults'), true)}
                                    </Link>
                                </span>
                            </div>
                            <div className='s-block__sub'>{i18n('Profile2.addresses.subs.defaults')}</div>
                        </div>
                    )}
                    {settings.tld === 'ru' && !delivery.length && (
                        <div className={b('block')}>
                            <div className='s-block__sub s-block__sub_hl'>
                                <span onClick={methods.addDelivery} data-t='pssp:add-delivery'>
                                    <Link href={deLinkParams.url} onClick={deLinkParams.onClick}>
                                        {this.getLinkText(i18n('Profile.addresses.add.delivery'), true)}
                                    </Link>
                                </span>
                            </div>
                            <div className='s-block__sub'>{i18n('Profile2.addresses.subs.delivery')}</div>
                        </div>
                    )}
                    {settings.tld === 'ru' && delivery.length > 0 && (
                        <div className={b('block')}>
                            <Toggle
                                dt='section:delivery'
                                cls={b('delivery')}
                                margin={5}
                                alwaysShown={MAX_DELIVERY_SHOWN}
                                itemClassName='s-block__sub s-block__sub_hl'
                                itemOnClick={methods.editDelivery}
                                title={i18n('Profile.addresses.type.delivery')}
                                toggleText={toggleText}
                                items={items}
                            />
                        </div>
                    )}
                </div>
                <div className={b('map')}>
                    <Map className={b('yamap')} points={mapAddrs} settings={settings} width={650} height={225} />
                </div>
                <Modal
                    visible={common.edit === EDIT_MODE_ADDRESSES || common.edit === EDIT_MODE_DELIVERY_ADDRESSES}
                    zIndexGroupLevel={-0.5}
                    dataT='addresses-block'
                >
                    <AddressesControl
                        modal={true}
                        settings={settings}
                        addresses={addresses}
                        dispatch={dispatch}
                        common={common}
                        edit={common.edit}
                    />
                </Modal>
            </Section>
        );
    }
}

AddressesBlock.propTypes = {
    settings: PropTypes.object.isRequired,
    dispatch: PropTypes.func.isRequired,
    addresses: PropTypes.object.isRequired,
    common: PropTypes.object.isRequired,
    methods: PropTypes.object.isRequired
};

export default withMethods(AddressesBlock, metrics.goal.bind(metrics), {
    addNavi: `${ADDRESSES_GOAL_PREFIX}_add_navi`,
    editNavi: `${ADDRESSES_GOAL_PREFIX}_edit_navi`,
    addDelivery: `${ADDRESSES_GOAL_PREFIX}_add_delivery`,
    editDelivery: `${ADDRESSES_GOAL_PREFIX}_edit_delivery`
});
