import React from 'react';
import PropTypes from 'prop-types';

import {Link} from '@components/Link';
import Modal from '@components/Wrappers/Modal.jsx';

import {AppPasswordsList} from '@blocks/morda/app_passwords/app_passwords_list.jsx';

import metrics from '@blocks/metrics';

import * as extracted from './passwords.js';
import {setEditMode} from '@blocks/common/actions';
import {showDisableScreen} from '@blocks/morda/app_passwords/actions';

export default class Passwords extends React.PureComponent {
    static sendMetrics() {
        metrics.send(['Управление доступом', 'Включить пароли приложений: ']);
    }

    constructor(props) {
        super(props);

        this.displayName = 'app-passwords';
        this.showAppPasswordsModal = extracted.showAppPasswordsModal.bind(this);
        this.onLinkClick = extracted.onLinkClick.bind(this);
        this.redirectToAppPasswordsList = extracted.redirectToAppPasswordsList.bind(this);
    }

    redirectTo = (url) => {
        const {retpath} = this.props;

        let redirectUrl = url;

        if (retpath) {
            redirectUrl += `?retpath=${encodeURIComponent(retpath)}`;
        }

        window.location.href = redirectUrl;
    };

    closeModal = () => {
        const {settings = {}, dispatch, common} = this.props;
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;
        const isStandAlonePage = common && common.currentPage !== '/profile';

        if (isPhone || isStandAlonePage) {
            if (typeof window !== 'undefined') {
                // using push('/profile') causes a bug when we get a /profile/personal-info page instead of /profile
                this.redirectTo('/profile');
            }
        } else {
            dispatch(setEditMode(null));
            dispatch(showDisableScreen(false));
        }
    };

    render() {
        const {edit, dispatch, settings, isEnabled, appPasswords, is2faEnabled} = this.props;

        return (
            <div data-t={`pssp:passwords_${isEnabled ? 'enabled' : 'disabled'}`}>
                <span onClick={Passwords.sendMetrics}>
                    <Link pseudo={true} onClick={isEnabled ? this.showAppPasswordsModal : this.onLinkClick}>
                        {isEnabled ? i18n('Profile.apppasswords.main.title') : i18n('Profile.apppasswords.enable')}
                    </Link>
                </span>
                {!settings.isTouch && (
                    <Modal
                        visible={edit === 'apppasswords-list'}
                        zIndexGroupLevel={-0.5}
                        onCloserClick={this.closeModal}
                        dataT='apppasswords'
                    >
                        <AppPasswordsList
                            closeModal={this.closeModal}
                            isEnabled={isEnabled}
                            edit={edit}
                            dispatch={dispatch}
                            settings={settings}
                            is2faEnabled={is2faEnabled}
                            appPasswords={appPasswords}
                        />
                    </Modal>
                )}
            </div>
        );
    }
}

Passwords.propTypes = {
    edit: PropTypes.string,
    dispatch: PropTypes.func.isRequired,
    settings: PropTypes.object.isRequired,
    common: PropTypes.object,
    retpath: PropTypes.string.isRequired,
    isEnabled: PropTypes.bool.isRequired,
    appPasswords: PropTypes.object.isRequired,
    is2faEnabled: PropTypes.bool.isRequired,
    passwordStrength: PropTypes.number.isRequired
};
