import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Button} from '@components/Button';
import {Modal} from '@components/Modal';
import {PhoneConfirmation} from '@screens/PhoneConfirmation';
import {ChromeIcon, DesktopIcon, MobileIcon, PlusIcon, SafariIcon, TabletIcon, TrashIcon} from './icons';
import './Webauthn.styl';

const b = cn('Webauthn');

const DEVICE_ICON_COMPONENTS = {
    tablet: <TabletIcon />,
    mobile: <MobileIcon />,
    desktop: <DesktopIcon />
};

const BROWSER_ICON_COMPONENTS = {
    Chrome: <ChromeIcon />,
    ChromeMobile: <ChromeIcon />,
    Safari: <SafariIcon />,
    SafariMobile: <SafariIcon />
};

export class Webauthn extends React.Component {
    state = {isPhoneModalOpened: false};
    savedAction = () => null;
    isPhoneConfirmInProccess = false;
    componentDidMount() {
        const {fetchWebauthnCredentials} = this.props;

        fetchWebauthnCredentials();
    }
    componentDidUpdate({error: oldError}) {
        const {error} = this.props;

        if (oldError !== error) {
            if (error === 'phone.not_confirmed') {
                this.confirmPhone();
            } else if (oldError === 'phone.not_confirmed') {
                this.savedAction();
            }
        }

        return true;
    }
    confirmPhone = () => {
        const {confirmPhone} = this.props;

        confirmPhone().done(() => {
            this.isPhoneConfirmInProccess = true;
            this.openModal();
        });
    };
    commitPhone = () => {
        const {commitPhone, removeError} = this.props;

        commitPhone().done(() => {
            this.isPhoneConfirmInProccess = false;
            this.closeModal();
            removeError();
        });
    };
    createCred = () => {
        const {loading = false, postWebauthnReg} = this.props;

        if (!loading) {
            this.savedAction = () => postWebauthnReg();
            this.isPhoneConfirmInProccess ? this.openModal() : postWebauthnReg();
        }
    };
    removeCred = (e) => {
        const {loading = false, removeCred, list} = this.props;
        const {index} = e.currentTarget.dataset;
        const {extId} = list[Number(index)];

        if (!loading) {
            this.savedAction = () => removeCred(extId);
            removeCred(extId);
        }
    };
    closeModal = () => this.setState({isPhoneModalOpened: false});
    openModal = () => this.setState({isPhoneModalOpened: true});
    render() {
        const {list = [], isNavigatorError = false, hasSuggestedDevice = false, hasSecurePhone = false} = this.props;
        const {isPhoneModalOpened} = this.state;
        const showButton = hasSecurePhone && !isNavigatorError && !hasSuggestedDevice;

        return (
            <div className={b()}>
                {showButton && (
                    <div className={b('button')}>
                        <Button
                            text={i18n('Frontend.webauthn_login.button')}
                            iconLeft={PlusIcon}
                            view='action'
                            width='max'
                            size='l'
                            onClick={this.createCred}
                        />
                    </div>
                )}
                <div className={b('list')}>
                    {list.map((cred, index) => (
                        <div className={b('cred')} key={cred.extId}>
                            <div className={b('pic', {[cred.deviceType]: true})}>
                                {DEVICE_ICON_COMPONENTS[cred.deviceType]}
                                <div className={b('browserIcon')}>{BROWSER_ICON_COMPONENTS[cred.browser]}</div>
                            </div>
                            <div className={b('info')}>
                                <div className={b('deviceName')}>{cred.deviceName || cred.os}</div>
                                {cred.isSuggested ? (
                                    <pre className={b('browser')}>
                                        {cred.browser} •{' '}
                                        <span className={b('suggested')}>
                                            {i18n('Frontend.webauthn_login.suggested')}
                                        </span>
                                    </pre>
                                ) : (
                                    <div className={b('browser')}>{cred.browser}</div>
                                )}
                            </div>
                            <div className={b('trash')} data-index={index} onClick={this.removeCred}>
                                <TrashIcon />
                            </div>
                        </div>
                    ))}
                </div>
                <Modal visible={isPhoneModalOpened} onCloserClick={this.closeModal}>
                    <div className={b('modal')}>
                        <PhoneConfirmation
                            onConfirm={this.commitPhone}
                            onRetry={this.confirmPhone}
                            place='without_password'
                        />
                    </div>
                </Modal>
            </div>
        );
    }
}

Webauthn.propTypes = {
    list: PropTypes.arrayOf(
        PropTypes.shape({
            extId: PropTypes.string.isRequired,
            deviceType: PropTypes.string.isRequired,
            browser: PropTypes.string.isRequired,
            deviceName: PropTypes.string,
            os: PropTypes.string.isRequired,
            isSuggested: PropTypes.bool.isRequired
        })
    ),
    error: PropTypes.string,
    hasSecurePhone: PropTypes.bool,
    loading: PropTypes.bool,
    isNavigatorError: PropTypes.bool,
    hasSuggestedDevice: PropTypes.bool,
    confirmPhone: PropTypes.func.isRequired,
    commitPhone: PropTypes.func.isRequired,
    removeError: PropTypes.func.isRequired,
    fetchWebauthnCredentials: PropTypes.func.isRequired,
    postWebauthnReg: PropTypes.func.isRequired,
    removeCred: PropTypes.func.isRequired
};
