import api from '@blocks/api';
import {updateErrors} from '@blocks/actions/form';
import {FIELDS_NAMES} from '@components/Field/names';
import {getError} from '@blocks/utils';
import {getTrackId} from '@blocks/selectors';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {saveActionForRepeat} from '@blocks/common/actions';
import {
    setPhoneConfirmationDenyResendUntilTime,
    updatePhoneNumber,
    setPhoneConfirmationInternationalPhoneNumber
} from '@blocks/actions/phoneConfirm';
import {preformatMakeCredReq, publicKeyCredentialToJSON} from './base64url-arraybuffer';
import {setWebauthnLoading, setWebauthnError, setWebauthnList} from './actions';

const makeRequest = (dispatch, method, data = {}) => {
    dispatch(setWebauthnLoading(true));
    return api
        .request(method, data)
        .fail(({code}) =>
            ['phone.not_found'].includes(code) ? window.location.reload() : dispatch(setWebauthnError(code))
        )
        .always(() => dispatch(setWebauthnLoading(false)));
};

export const removeError = () => (dispatch) => dispatch(setWebauthnError(null));

export const fetchWebauthnList = (dispatch) =>
    makeRequest(dispatch, '/auth/webauthn/list').done(({list}) => dispatch(setWebauthnList(list)));

export const fetchWebauthnCredentials = () => (dispatch) => fetchWebauthnList(dispatch);

export const postWebauthnReg = () => (dispatch, getState) => {
    const trackId = getTrackId(getState(), TRACK_TYPES.COMMON);

    dispatch(saveActionForRepeat(postWebauthnReg));

    try {
        if (
            !navigator ||
            !navigator.credentials ||
            !navigator.credentials.create ||
            typeof navigator.credentials.create !== 'function'
        ) {
            throw Error;
        }
    } catch (err) {
        dispatch(setWebauthnError('navigator.credentials'));
        api.log(`Webauthn Error: browser dont have navigator.credentials.create api | trackId: ${trackId} | code: 1`);
        return;
    }

    return makeRequest(dispatch, '/auth/webauthn/reg/submit')
        .done(async ({makeCredOptions = {}}) => {
            const publicKey = preformatMakeCredReq(makeCredOptions);

            try {
                const response2 = await navigator.credentials.create({publicKey});
                const assertion = publicKeyCredentialToJSON(response2);

                while (assertion.response.attestationObject.length % 4 !== 0) {
                    assertion.response.attestationObject += '=';
                }

                while (assertion.response.clientDataJSON.length % 4 !== 0) {
                    assertion.response.clientDataJSON += '=';
                }

                makeRequest(dispatch, '/auth/webauthn/reg/commit', {
                    attestation: assertion.response.attestationObject,
                    client: assertion.response.clientDataJSON
                })
                    .done(() => fetchWebauthnList(dispatch))
                    .fail(({code}) => api.log(`Webauthn Error: ${code} | trackId: ${trackId} | code: 2`));
            } catch (err) {
                dispatch(setWebauthnError('navigator.credentials'));
                api.log(`Webauthn Error: ${err.message} | trackId: ${trackId} | code: 3`);
            }
        })
        .fail(({code}) => {
            if (code !== 'phone.not_confirmed') {
                api.log(`Webauthn Error: ${code} | trackId: ${trackId} | code: 4`);
            }
        });
};

export const confirmPhone = () => (dispatch, getState) => {
    const {settings: {language = 'ru'} = {}, phones: {restore: [{id} = {}] = []} = {}} = getState();

    dispatch(saveActionForRepeat(confirmPhone));

    return api
        .request('phone-confirm-code-submit', {phone_id: id, display_language: language})
        .done(({deny_resend_until: denyResendUntil, number: {masked_international: phone}}) => {
            dispatch(setPhoneConfirmationDenyResendUntilTime(denyResendUntil));
            dispatch(updatePhoneNumber(phone));
            dispatch(setPhoneConfirmationInternationalPhoneNumber(phone));
        });
};

export const commitPhone = () => (dispatch, getState) => {
    const {form: {values: {phoneCode: code = ''} = {}} = {}} = getState();

    dispatch(saveActionForRepeat(commitPhone));

    return api.request('phone-confirm-code', {code: code ? code.trim() : ''}).fail((error = {}) => {
        const errorCode = Array.isArray(error.errors) ? error.errors[0] : 'global';

        dispatch(updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, errorCode)}));
    });
};

export const removeCred = (extId) => (dispatch) => {
    dispatch(saveActionForRepeat(removeCred, extId));

    return api.request('/auth/webauthn/remove', {extId}).done(() => fetchWebauthnList(dispatch));
};
