import React, {useState, useCallback} from 'react';
import PropTypes from 'prop-types';
import {push} from 'react-router-redux';
import Modal from '@components/Wrappers/Modal.jsx';
import {Link} from '@components/Link';
import metrics from '@blocks/metrics';
import {EnterWithoutPasswordBody} from './EnterWithoutPasswordBody';

const sendMetrics = () => metrics.send(['Безопасность аккаунта', 'Вход без пароля']);

const useModalControl = () => {
    const [isModalOpened, toggleWithoutPasswordPopup] = useState(false);

    const openWithoutPasswordPopup = () => toggleWithoutPasswordPopup(true);
    const closeWithoutPasswordPopup = () => toggleWithoutPasswordPopup(false);

    return [isModalOpened, openWithoutPasswordPopup, closeWithoutPasswordPopup];
};

const EnterWithoutPassword = ({
    isStrongPolicyOn,
    loginOptions,
    phoneNumber,
    hasPhone,
    hasAliasPhone,
    dispatch,
    isPhone
}) => {
    const [isModalOpened, openWithoutPasswordPopup, closeWithoutPasswordPopup] = useModalControl();
    const openWithoutPasswordPage = useCallback(() => dispatch(push('/profile/auth-settings')), []);

    return (
        <div data-t='pssp:without_password'>
            <div className='s-block__sub s-block__sub_hl'>
                <span onClick={sendMetrics}>
                    <Link pseudo={true} onClick={isPhone ? openWithoutPasswordPage : openWithoutPasswordPopup}>
                        {i18n('Frontend.magic_auth.title')}
                    </Link>
                </span>
                <Modal
                    visible={isModalOpened}
                    zIndexGroupLevel={-0.5}
                    onCloserClick={closeWithoutPasswordPopup}
                    dataT='without-password'
                >
                    <div data-t='pssp:without_password-modal'>
                        <EnterWithoutPasswordBody
                            isStrongPolicyOn={isStrongPolicyOn}
                            dispatch={dispatch}
                            phoneNumber={phoneNumber}
                            hasPhone={hasPhone}
                            hasAliasPhone={hasAliasPhone}
                            loginOptions={loginOptions}
                        />
                    </div>
                </Modal>
            </div>
            <div className='s-block__sub'>{i18n('Frontend.magic_auth.subtitle')}</div>
        </div>
    );
};

const EnterWithoutPasswordPage = (props) => {
    const {dispatch, person = {}, access = {}, phones} = props;
    const hasAliasPhone = ((phones && phones.restore) || []).some((phone) => phone.isAlias);
    const isStrongPolicyOn = (access.passwordInfo || {}).strong_policy_on;
    const phoneNumber = ((phones && phones.restore).find((item) => item.isDefault) || {}).number;
    const hasPhone = phones && phones.restore.length;

    return (
        <div className='p-control-form'>
            <EnterWithoutPasswordBody
                dispatch={dispatch}
                phoneNumber={phoneNumber}
                hasPhone={hasPhone}
                place='page'
                loginOptions={person.loginOptions}
                hasAliasPhone={hasAliasPhone}
                isStrongPolicyOn={isStrongPolicyOn}
            />
        </div>
    );
};

EnterWithoutPassword.propTypes = {
    loginOptions: PropTypes.shape({
        qrCodeLoginForbidden: PropTypes.bool.isRequired,
        smsCodeLoginForbidden: PropTypes.bool
    }),
    isPhone: PropTypes.bool.isRequired,
    dispatch: PropTypes.func.isRequired,
    hasAliasPhone: PropTypes.bool,
    isStrongPolicyOn: PropTypes.bool,
    phoneNumber: PropTypes.string,
    hasPhone: PropTypes.bool
};

EnterWithoutPasswordPage.propTypes = {
    person: PropTypes.object.isRequired,
    dispatch: PropTypes.func.isRequired,
    phones: PropTypes.shape({
        restore: PropTypes.array,
        other: PropTypes.array
    }),
    access: PropTypes.object,
    common: PropTypes.object,
    isStrongPolicyOn: PropTypes.bool
};

export {EnterWithoutPasswordPage, EnterWithoutPasswordBody, EnterWithoutPassword};
