import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Link} from '@components/Link';

import Modal from '@components/Wrappers/Modal.jsx';
import {Popup} from '@components/Popup';
import {Spin} from '@components/Spin';
import {YaIconRu} from './Icons/YaIconRu';
import {YaIconEn} from './Icons/YaIconEn';
import {LockIcon} from './Icons/LockIcon';
import {PayIcon} from './Icons/PayIcon';

import {getCardsInfo, resetCardBindingForm} from '@blocks/morda/cards';
import {setEditMode} from '@blocks/common/actions';

import CardsControl from '@blocks/morda/cards/cards_control.jsx';
import {Section} from '@blocks/morda/new/section/section.jsx';
import Scroll from '@blocks/morda/new/scroll/scroll.jsx';
import BankCard from './bank_card.jsx';
import {selectCard, setErrors} from '@blocks/morda/cards/actions';
import {FamilyPaySettingsWidget} from '@blocks/morda/new/family/widgets/FamilyPaySettings';
import * as extracted from './bank_cards_extracted';

const b = cn('BankCards');

export default class BankCards extends React.PureComponent {
    familyCardRef = React.createRef(null);
    state = {isFamilyPopupOpened: false};
    bindCard = (e) => extracted.bindCard.call(this, e);
    componentDidMount() {
        const {billing} = this.props;
        const hasCards = billing.hasOwnProperty('cards');
        const hasErrors = billing.hasOwnProperty('errors');

        if (!hasCards || (hasErrors && billing.errors.length)) {
            this.props.dispatch(getCardsInfo());
        }
    }

    renderAddBankCard = (isDisabled) => {
        const {settings: {isTouch = false} = {}} = this.props;

        return <BankCard isTouch={isTouch} isAdd={true} isDisabled={isDisabled} onClick={this.bindCard} />;
    };

    renderTitle = () => {
        const {tld} = this.props;
        const isRu = ['ru', 'ua'].includes(tld);

        return (
            <div className={b('title')}>
                <span>{i18n('Profile2.titles.bank_cards_v2')}</span>
                <span className={b('yaLogoWrapper')}>
                    {isRu ? <YaIconRu className={b('yaLogo')} /> : <YaIconEn className={b('yaLogo')} />}
                </span>
                <span className={b('payIconWrapper')}>
                    <PayIcon className={b('payIcon')} />
                </span>
            </div>
        );
    };

    closeModal = () => {
        const {dispatch} = this.props;

        dispatch(setErrors([]));
        dispatch(setEditMode(''));
        dispatch(selectCard(''));
        dispatch(resetCardBindingForm());
    };

    openFamilyPopup = () => this.setState({isFamilyPopupOpened: true});
    toggleFamilyPopup = () => this.setState({isFamilyPopupOpened: !this.state.isFamilyPopupOpened});
    closeFamilyPopup = () => this.setState({isFamilyPopupOpened: false});

    render() {
        const {
            billing,
            edit,
            dispatch,
            settings,
            retpath,
            isOrderHistoryExp,
            v2,
            yaPayLanding,
            uid,
            hasRestorePhone
        } = this.props;
        const {isFamilyPopupOpened} = this.state;
        const scrollItems = [];
        const hasCards = billing.hasOwnProperty('cards') && billing.cards.length > 0;
        const hasErrors = billing.hasOwnProperty('errors') && billing.errors.length > 0;

        if (hasCards) {
            for (let i = 0; i < billing.cards.length; ++i) {
                const {isFamilyCard, familyAdminUid, proto} = billing.cards[i];

                const isFamilyAdmin = isFamilyCard && familyAdminUid === uid;
                const isFamilyMemberWithPhone = isFamilyCard && !isFamilyAdmin && hasRestorePhone;
                const showCard = isFamilyCard
                    ? isFamilyAdmin || isFamilyMemberWithPhone
                    : proto !== 'yamoney_walletapi';

                if (showCard) {
                    scrollItems.push(
                        <BankCard
                            v2={v2}
                            key={billing.cards[i].number}
                            numberText={isFamilyMemberWithPhone ? i18n('Profile2.bank_cards.family') : ''}
                            card={billing.cards[i]}
                            dispatch={dispatch}
                            isTouch={settings.isTouch}
                            cardRef={isFamilyMemberWithPhone ? this.familyCardRef : undefined}
                            onClick={isFamilyMemberWithPhone ? this.toggleFamilyPopup : undefined}
                        />
                    );
                }
            }
        }

        const cardsLength = billing.cards
            ? billing.cards.filter((card) => card.proto === 'trust' || card.proto === 'fake').length
            : 0;

        scrollItems.push(this.renderAddBankCard(cardsLength > 4));

        return (
            <Section className={b()} icon='card' title={this.renderTitle()}>
                <div className={b('description')}>
                    <span className={b('descriptionText')}>{i18n('Profile2.bank_cards.description')}</span>
                    <Link theme='ghost' view='ghost' href={yaPayLanding} className={b('descriptionLink')}>
                        {i18n('Profile2.bank_cards.description-link')}
                    </Link>
                </div>
                {hasCards && !hasErrors && (
                    <Scroll
                        isTouch={settings.isTouch}
                        items={scrollItems}
                        fixed={true}
                        sizes={[
                            {
                                maxWidth: 0,
                                cardWidth: 200
                            }
                        ]}
                        marginRight={6}
                    />
                )}
                {billing.progress && (
                    <div className={b('loading')}>
                        <Spin progress={true} size='m' />
                    </div>
                )}
                {(!hasCards || hasErrors) && !billing.progress && (
                    <Scroll
                        isTouch={settings.isTouch}
                        items={scrollItems}
                        fixed={true}
                        sizes={[
                            {
                                maxWidth: 0,
                                cardWidth: 200
                            }
                        ]}
                        marginRight={6}
                    />
                )}
                {!settings.isTouch && (
                    <Modal
                        visible={edit === 'cards'}
                        zIndexGroupLevel={-0.5}
                        onCloserClick={this.closeModal}
                        dataT='bank-cards'
                    >
                        <CardsControl
                            retpath={retpath}
                            billing={billing}
                            dispatch={dispatch}
                            settings={settings}
                            modal={true}
                        />
                    </Modal>
                )}
                <div className={b('safeInfo')}>
                    <span className={b('safeInfoIconWrapper')}>
                        <LockIcon />
                    </span>
                    <span className={b('safeInfoText')}>{i18n('Profile2.bank_cards.safe-info-text')}</span>
                </div>

                {isOrderHistoryExp && v2 && (
                    <Link href='/order-history' className={b('link')}>
                        {i18n('Profile2.bank_cards.order_history')}
                    </Link>
                )}
                <Popup
                    hasTail={false}
                    directions={[
                        'right-top',
                        'left-top',
                        'bottom-center',
                        'bottom-start',
                        'bottom-end',
                        'top-center',
                        'top-start',
                        'top-end',
                        'bottom-start'
                    ]}
                    target='anchor'
                    mainOffset={8}
                    anchor={this.familyCardRef}
                    visible={isFamilyPopupOpened}
                    onClose={this.closeFamilyPopup}
                    zIndex={500}
                >
                    {isFamilyPopupOpened && <FamilyPaySettingsWidget />}
                </Popup>
            </Section>
        );
    }
}

BankCards.propTypes = {
    dispatch: PropTypes.func.isRequired,
    v2: PropTypes.bool,
    uid: PropTypes.number.isRequired,
    hasRestorePhone: PropTypes.bool.isRequired,
    isOrderHistoryExp: PropTypes.bool.isRequired,
    billing: PropTypes.shape({
        cards: PropTypes.array,
        progress: PropTypes.bool,
        errors: PropTypes.array
    }).isRequired,
    retpath: PropTypes.string,
    settings: PropTypes.object.isRequired,
    edit: PropTypes.string,
    tld: PropTypes.string.isRequired,
    yaPayLanding: PropTypes.string
};
