import React from 'react';
import PropTypes from 'prop-types';
import {Link} from '@components/Link';
import {Popup} from '@components/Popup';
import {FamilyCard} from '../../components/FamilyCard';
import {FamilyPaySettings} from '../../components/FamilyPaySettings';
import {FamilyUniversalTile} from '../../components/FamilyUniversalTile';
import {metrics} from '../../utils/metrics';
import {addCurrencySymbol} from '../../utils/currency';
import {CARDS, LIMIT_MODES, PAGES} from '../../const';

const STATES = {
    SETTINGS: 'settings',
    CARDS: 'cards',
    PHONE: 'phone',
    HIDE: 'hide'
};

const getActiveTitle = (isAdminAccess, isCardActualyEnabled, balance, limitMode, cardInfo) => {
    if (isAdminAccess) {
        if (cardInfo && cardInfo.bound) {
            const hasDots = cardInfo.maskedNumber.length === 14;
            const number = hasDots ? cardInfo.maskedNumber.slice(-4) : cardInfo.maskedNumber;

            return `${i18n('Family.cards.pay.hasCard.title')} ${hasDots ? '••' : ''} ${number}`;
        }

        return i18n('Family.cards.pay.hasCard.title');
    }

    if (isCardActualyEnabled) {
        return limitMode === LIMIT_MODES.NOLIMIT
            ? i18n('Family.cards.pay.hasCard.title.member.nolimit')
            : i18n('Family.cards.pay.hasCard.title.member.enabled').replace('%balance', balance);
    }

    return i18n('Family.cards.pay.hasCard.title.member.disabled');
};
const getActiveDescription = (isAdminAccess, isCardActualyEnabled) => {
    if (isAdminAccess) {
        return i18n('Family.cards.pay.hasCard.description');
    }

    return isCardActualyEnabled
        ? i18n('Family.cards.pay.hasCard.description.member.enabled')
        : i18n('Family.cards.pay.hasCard.description.member.disabled');
};

export class FamilyCardPay extends React.Component {
    cardRef = React.createRef(null);
    goToCardsPage = () => this.props.goToPage(PAGES.payCards);
    onGoToCardsClick = () => {
        metrics.sendPay(['Нажатие на баннер', STATES.CARDS]);
        metrics.goalPay('add_card_button');
        this.goToCardsPage();
    };
    onTileClick = () => {
        const {setPaySettingsOpenedState, isPaySettingsOpened, loading} = this.props;

        metrics.sendPay(['Нажатие на баннер', STATES.SETTINGS]);

        if (!loading) {
            setPaySettingsOpenedState(!isPaySettingsOpened);
        }
    };
    phoneBindOnSuccess = () => {
        const {fetchFamily, isAdminAccess} = this.props;

        metrics.sendPay(['Добавление телефона', 'Успех']);
        fetchFamily().done(() => isAdminAccess && this.goToCardsPage());
    };
    phoneBindOnClose = () => {
        metrics.sendPay(['Добавление телефона', 'Закрытие']);
    };
    goToAddPhonePage = () => {
        metrics.sendPay(['Нажатие на баннер', STATES.PHONE]);
        this.props.goToPage(PAGES.addPhone);
    };
    onPopupClose = () => this.props.setPaySettingsOpenedState(false);
    getState = () => {
        const {cardInfo: {cardId, bound} = {}, isAbleToUsePay, isFamilyPayExp} = this.props;

        if (!isFamilyPayExp) {
            return STATES.HIDE;
        }

        if (isAbleToUsePay) {
            return bound && cardId ? STATES.SETTINGS : STATES.CARDS;
        }

        return STATES.PHONE;
    };
    render() {
        const {
            currency,
            balance,
            limitMode,
            language,
            isPaySettingsOpened,
            isAdminAccess,
            isCardActualyEnabled,
            cardInfo
        } = this.props;
        const balanceStr = addCurrencySymbol(balance, currency, language);
        const state = this.getState();

        if (state === STATES.HIDE) {
            return null;
        }

        if (state === STATES.SETTINGS) {
            return (
                <>
                    <FamilyCard type={CARDS.pay} tile={true} innerRef={this.cardRef} onClick={this.onTileClick}>
                        <FamilyUniversalTile
                            type='block'
                            verticalAlignCenter={true}
                            title={getActiveTitle(isAdminAccess, isCardActualyEnabled, balanceStr, limitMode, cardInfo)}
                            description={getActiveDescription(isAdminAccess, isCardActualyEnabled)}
                            iconLeft='card'
                            iconRight='3points'
                        />
                    </FamilyCard>
                    <Popup
                        hasTail={false}
                        directions={['bottom-right']}
                        target='anchor'
                        mainOffset={8}
                        anchor={this.cardRef}
                        visible={isPaySettingsOpened}
                        onClose={this.onPopupClose}
                        zIndex={500}
                    >
                        <FamilyPaySettings />
                    </Popup>
                </>
            );
        }
        if (state === STATES.CARDS) {
            return (
                <FamilyCard type={CARDS.pay} tile={true} onClick={this.onGoToCardsClick}>
                    <FamilyUniversalTile
                        type='block'
                        verticalAlignCenter={true}
                        tile={true}
                        title={i18n('Family.cards.pay.title')}
                        description={<Link pseudo={true}>{i18n('Family.cards.pay.text')}</Link>}
                        iconLeft='card'
                        iconRight='arrowRight'
                    />
                </FamilyCard>
            );
        }
        if (state === STATES.PHONE) {
            return (
                <>
                    <FamilyCard
                        type={CARDS.pay}
                        small={true}
                        bg='multicolored'
                        textIcon='phone'
                        textIconRight='arrowRight'
                        title={i18n('Family.cards.pay.noSecure.title')}
                        text={i18n('Family.cards.pay.noSecure.text')}
                        onClick={this.goToAddPhonePage}
                    />
                </>
            );
        }
    }
}

FamilyCardPay.propTypes = {
    phonesLink: PropTypes.string,
    currency: PropTypes.string,
    language: PropTypes.string,
    loading: PropTypes.bool,
    hasFamily: PropTypes.bool,
    isAdminAccess: PropTypes.bool,
    isPaySettingsOpened: PropTypes.bool,
    isAbleToUsePay: PropTypes.bool,
    isCardActualyEnabled: PropTypes.bool,
    isFamilyPayExp: PropTypes.bool,
    balance: PropTypes.number,
    limitMode: PropTypes.oneOf(Object.values(LIMIT_MODES)),
    cardInfo: PropTypes.shape({
        cardId: PropTypes.string,
        bound: PropTypes.bool
    }),
    setPaySettingsOpenedState: PropTypes.func.isRequired,
    fetchFamily: PropTypes.func.isRequired,
    goToPage: PropTypes.func.isRequired,
    isEnabled: PropTypes.bool
};
