import React from 'react';
import PropTypes from 'prop-types';
import {Field} from '@components/Field';
import {RadioButton} from '@components/RadioButton';
import {BirthdayField} from '@components/BirthdayField';
import {cn} from '@bem-react/classname';
import {FamilyAvatar} from '../FamilyAvatar';
import {KIDDISH_INPUT_FIELDS} from '../../const';
import './FamilyKiddishInput.styl';

const b = cn('FamilyKiddishInput');

export class FamilyKiddishInput extends React.Component {
    componentWillUnmount = () => {
        const {setKiddishFieldError, field} = this.props;

        setKiddishFieldError({field, error: ''});
    };
    onChangeEvent = (e) => {
        if (e && e.target) {
            this.onChangeValue(e.target.value);
        }
    };
    onChangeValue = (value) => {
        const {field, updateKiddish} = this.props;

        updateKiddish({[field]: value});
    };
    onChangeAvatar = (e) => {
        if (e && e.currentTarget) {
            this.onChangeValue(e.currentTarget.dataset.avatar);
        }
    };
    setError = (error = '') => {
        const {setKiddishFieldError, field} = this.props;

        setKiddishFieldError({field, error});
    };
    renderInput = () => {
        const {field, value, error, text, options, placeholders, isTouch, kiddishAvatars, setInProgress} = this.props;

        switch (field) {
            case KIDDISH_INPUT_FIELDS.AVATAR:
                return (
                    <div className={b('avatars')}>
                        {kiddishAvatars.map((avatar) => (
                            <div
                                className={b('avatar', {isCurrent: value === avatar})}
                                key={avatar}
                                onClick={this.onChangeAvatar}
                                data-avatar={avatar}
                            >
                                <FamilyAvatar slot={{isKiddish: true, avatar}} size='m' isTouch={isTouch} />
                            </div>
                        ))}
                    </div>
                );
            case KIDDISH_INPUT_FIELDS.NAME:
                return (
                    <Field
                        size='l'
                        label={text}
                        value={value}
                        name='displayName'
                        type='text'
                        onChange={this.onChangeValue}
                        view='floating-label'
                    />
                );
            case KIDDISH_INPUT_FIELDS.GENDER:
                return (
                    <>
                        <div className={b('label')}>{text}</div>
                        <div className={b('radio')}>
                            <RadioButton
                                width='max'
                                value={value}
                                options={options.map(({text, value, checked}) => ({
                                    children: text,
                                    value,
                                    checked,
                                    'data-t': `kiddish:gender:${value || 'unknow'}`
                                }))}
                                onChange={this.onChangeEvent}
                            />
                        </div>
                    </>
                );
            case KIDDISH_INPUT_FIELDS.BIRTHDAY:
                return (
                    <>
                        <div className={b('label')}>{text}</div>
                        <BirthdayField
                            onChange={this.onChangeValue}
                            setError={this.setError}
                            value={value}
                            dayPlaceholder={placeholders.day}
                            yearPlaceholder={placeholders.year}
                            setInProgress={setInProgress}
                            type='kiddish'
                            error={error}
                        />

                        {error && <div className={b('error')}>{i18n('Frontend.birthday_errors_invalid')}</div>}
                    </>
                );
            default:
                return null;
        }
    };
    render() {
        const {field} = this.props;
        const input = this.renderInput();

        return input && <div className={b({field})}>{input}</div>;
    }
}

FamilyKiddishInput.propTypes = {
    type: PropTypes.string,
    field: PropTypes.string,
    value: PropTypes.string,
    error: PropTypes.string,
    text: PropTypes.string,
    isTouch: PropTypes.bool,
    options: PropTypes.array,
    kiddishAvatars: PropTypes.array.isRequired,
    placeholders: PropTypes.object,
    setInProgress: PropTypes.func,
    updateKiddish: PropTypes.func,
    setKiddishFieldError: PropTypes.func
};
