import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {redirectToRetpathFromWebview} from '@blocks/utils';
import {WEBVIEW_STATUS_SUCCESS, WEBVIEW_STATUS_CLOSE, WEBVIEW_STATUS_ERROR} from '@blocks/constants';
import {CloseButton} from '@blocks/pushes/components/CloseButton/CloseButton';
import {metrics} from '../../utils/metrics';
import {shortenLongNames} from '../../utils/shortenLongNames';
import {FamilyActionLayout} from '../../layouts/FamilyActionLayout';
import {FAMILY_ENTRY_PAGES_METRICS_MAP} from '../../const';
import {FamilyButton} from '../FamilyButton';
import {FamilyWarning} from '../FamilyWarning';
import {FamilyPayLimitsUsers} from './Users';
import {FamilyPayLimitsInput} from './Input';
import {FamilyPayLimitsControls} from './Controls';
import './FamilyPayLimits.styl';

const b = cn('FamilyPayLimits');

const LIMITS_ERRORS_STUBS = {
    taxi_familycard: {
        hasNotMembers: {
            title: i18n('Family.payLimits.stub.taxi.hasNotMembers'),
            decor: 'taxiLimitsHasNotMembers'
        },
        hasNotCard: {
            title: i18n('Family.payLimits.stub.taxi.hasNotCard'),
            decor: 'taxiLimitsHasNotCard'
        }
    }
};

export class FamilyPayLimits extends React.Component {
    componentDidMount() {
        const {isWebview} = this.props;

        metrics.sendPay([
            isWebview ? FAMILY_ENTRY_PAGES_METRICS_MAP.limitsWebview : FAMILY_ENTRY_PAGES_METRICS_MAP.limits,
            'Показ'
        ]);
    }
    componentWillUnmount() {
        const {resetPayLimitsForm, isWebview} = this.props;

        metrics.sendPay([
            isWebview ? FAMILY_ENTRY_PAGES_METRICS_MAP.limitsWebview : FAMILY_ENTRY_PAGES_METRICS_MAP.limits,
            'Закрытие'
        ]);

        resetPayLimitsForm();
    }
    onSaveClick = () => {
        const {
            goToMainPage,
            fetchFamily,
            postUsersLimits,
            setCurrentSlot,
            currentSlot,
            loading,
            setPaySettingsOpenedState,
            isLimitsPage,
            isAm,
            finishOkUrl,
            finishErrorUrl,
            origin,
            notRedirectWithoutRetpath,
            retpath
        } = this.props;

        metrics.sendPay(['Нажатие на кнопку Сохранить']);

        if (!loading) {
            postUsersLimits()
                .done(() => {
                    if (isLimitsPage) {
                        if (isAm) {
                            return (window.location.href = finishOkUrl);
                        } else if (origin && notRedirectWithoutRetpath) {
                            if (retpath) {
                                return redirectToRetpathFromWebview(retpath, WEBVIEW_STATUS_SUCCESS);
                            }
                            fetchFamily().done(() => {
                                setCurrentSlot(currentSlot);
                            });
                        } else {
                            // TODO
                        }
                    } else {
                        goToMainPage();
                        setPaySettingsOpenedState(false);
                        fetchFamily();
                    }
                })
                .fail(() => {
                    if (isLimitsPage) {
                        if (isAm) {
                            return (window.location.href = finishErrorUrl);
                        }

                        if (origin && notRedirectWithoutRetpath && retpath) {
                            return redirectToRetpathFromWebview(retpath, WEBVIEW_STATUS_ERROR);
                        }
                    }
                });
        }
    };
    onCloseButtonClick = () => {
        const {isAm, origin, retpath, finishCancelUrl} = this.props;

        if (isAm) {
            return (window.location.href = finishCancelUrl);
        }

        if (origin) {
            return redirectToRetpathFromWebview(retpath, WEBVIEW_STATUS_CLOSE);
        }
    };
    render() {
        const {
            isAm,
            isTouch,
            isLimitsPage,
            hideLimitsPageBackButton,
            currentSlot: {isAbleToUsePay = false, name = ''} = {},
            currentSlotFormLimitInfo: {isEnabled} = {},
            limitsPageBottomPadding = false,
            hasLimitsError,
            limitsError,
            origin
        } = this.props;
        const showWarning = !isAbleToUsePay && isEnabled;
        const hasCloseButton = isAm && isLimitsPage && !hideLimitsPageBackButton;

        if (hasLimitsError) {
            return (
                <div className={b({fixed: true})} data-t='family:pay-limits'>
                    <FamilyActionLayout
                        decor={LIMITS_ERRORS_STUBS[origin][limitsError].decor}
                        title={LIMITS_ERRORS_STUBS[origin][limitsError].title}
                        hideBackButton={true}
                    />
                </div>
            );
        }

        return (
            <div className={b({limitsPageBottomPadding, isTouch})} data-t='family:pay-limits'>
                {hasCloseButton && <CloseButton closeHandler={this.onCloseButtonClick} />}
                <div className={b('users', {noTouchNegativeMargin: isLimitsPage, isAm})}>
                    <FamilyPayLimitsUsers />
                </div>
                {showWarning && (
                    <div className={b('warning')}>
                        <FamilyWarning
                            text={i18n('Family.payLimits.warning').replace('%name', shortenLongNames(name))}
                        />
                    </div>
                )}
                <div className={b('input')}>
                    <FamilyPayLimitsInput />
                </div>
                <div className={b('controls')}>
                    <FamilyPayLimitsControls />
                </div>
                <div className={b('save')}>
                    <FamilyButton
                        view='action'
                        text={i18n('Frontend.save')}
                        onClick={this.onSaveClick}
                        dataT='family:pay-limits:save'
                    />
                </div>
            </div>
        );
    }
}

FamilyPayLimits.propTypes = {
    setPaySettingsOpenedState: PropTypes.func,
    goToMainPage: PropTypes.func,
    resetPayLimitsForm: PropTypes.func,
    postUsersLimits: PropTypes.func,
    fetchFamily: PropTypes.func,
    setCurrentSlot: PropTypes.func,
    isAm: PropTypes.bool,
    isTouch: PropTypes.bool,
    isLimitsPage: PropTypes.bool,
    loading: PropTypes.bool,
    showWarning: PropTypes.bool,
    currentSlot: PropTypes.shape({
        isAbleToUsePay: PropTypes.bool,
        name: PropTypes.string
    }),
    currentSlotFormLimitInfo: PropTypes.shape({
        isEnabled: PropTypes.bool
    }),
    finishOkUrl: PropTypes.string,
    finishCancelUrl: PropTypes.string,
    finishErrorUrl: PropTypes.string,
    retpath: PropTypes.string,
    origin: PropTypes.string,
    limitsPageBottomPadding: PropTypes.bool,
    notRedirectWithoutRetpath: PropTypes.bool,
    hideLimitsPageBackButton: PropTypes.bool,
    hasLimitsError: PropTypes.bool,
    limitsError: PropTypes.string,
    isWebview: PropTypes.bool
};
