import React from 'react';
import PropTypes from 'prop-types';
import debounce from 'lodash/debounce';
import {cn} from '@bem-react/classname';
import {Field} from '@components/Field';
import {Popup} from '@components/Popup';
import {metrics} from '../../../utils/metrics';
import {addCurrencySymbol} from '../../../utils/currency';
import {CURRENCIES, CURRENCIES_LIST, LIMIT_MODES, MAX_LIMIT} from '../../../const';
import {FamilyUniversalTileList} from '../../FamilyUniversalTileList';
import './FamilyPayLimitsInput.styl';

const b = cn('FamilyPayLimitsInput');

const FRAME_OPTIONS_TEXTS = {
    [LIMIT_MODES.DAY]: i18n('Family.payLimitsInput.frameOptions.day'),
    [LIMIT_MODES.WEEK]: i18n('Family.payLimitsInput.frameOptions.week'),
    [LIMIT_MODES.MONTH]: i18n('Family.payLimitsInput.frameOptions.month'),
    [LIMIT_MODES.TOTAL]: i18n('Family.payLimitsInput.frameOptions.max'),
    [LIMIT_MODES.NOLIMIT]: i18n('Family.payLimitsInput.frameOptions.nolimit')
};
const CURRENCIES_TEXTS = {
    [CURRENCIES.RUB]: i18n('Family.payLimitsInput.currencies.RUB'),
    [CURRENCIES.USD]: i18n('Family.payLimitsInput.currencies.USD'),
    [CURRENCIES.EUR]: i18n('Family.payLimitsInput.currencies.EUR'),
    [CURRENCIES.BYN]: i18n('Family.payLimitsInput.currencies.BYN'),
    [CURRENCIES.KZT]: i18n('Family.payLimitsInput.currencies.KZT'),
    [CURRENCIES.AMD]: i18n('Family.payLimitsInput.currencies.AMD'),
    [CURRENCIES.ILS]: i18n('Family.payLimitsInput.currencies.ILS')
};
const SUGGEST_VALUES = ['+500', '+1000', '+5000'];

const mapInputToNumber = (value) => Number(value.replace(/\D/g, '').replace(/^0+([1-9])/, (match, number) => number));

export class FamilyPayLimitsInput extends React.Component {
    constructor(props) {
        super(props);

        this.value = props.value;
    }
    onLimitModeClickFuncs = Object.keys(LIMIT_MODES).reduce((acc, limitMode) => {
        acc[limitMode] = () => {
            const {value, defaultValue, limitMode: prevLimitMode} = this.props;

            let updatedValue = null;

            if (prevLimitMode === 'NOLIMIT' && Number(value) === 0) {
                updatedValue = defaultValue;
            }

            metrics.sendPay(['Изменен тип лимита', limitMode]);
            this.updateLimits({limitMode, value: updatedValue || Math.min(value, MAX_LIMIT)});
            this.closeLimitModePopup();
        };
        return acc;
    }, {});
    mainRef = React.createRef();
    limitOptionsRef = React.createRef();
    currencyOptionsRef = React.createRef();
    state = {isLimitModeOpened: false, isCurrencyOptionsOpened: false, isLimitTypePopupOpened: false};
    componentDidUpdate(props) {
        const {currentSlot, value} = this.props;

        if (currentSlot !== props.currentSlot) {
            this.sendNewValueMetric.flush();
            this.value = value;
        }
    }
    componentWillUnmount() {
        this.sendNewValueMetric.flush();
    }
    updateLimits = (newLimits = {}) => {
        const {placeId, updatePayLimitForm, limitMode, value} = this.props;

        updatePayLimitForm({placeId, limit: {limitMode, value, ...newLimits}});
    };
    sendNewValueMetric = debounce((newValue) => {
        metrics.sendPay(['Изменение значение лимита', newValue > this.value ? 'Больше' : 'Меньше']);
        this.value = newValue;
    }, 1000);
    setValue = (input) => {
        const value = Math.min(mapInputToNumber(input), MAX_LIMIT);

        this.sendNewValueMetric(value);
        this.updateLimits({value});
    };
    onCurrencyOptionClick = (e) => {
        const {placeId, updatePayLimitForm} = this.props;
        const currency = e.currentTarget.dataset.value;

        metrics.sendPay(['Выбор другой валюты', currency]);
        updatePayLimitForm({placeId, currency});
        this.closeCurrencyPopup();
    };
    onInputChange = (value) => this.props.isEnabled && this.setValue(value);
    onValueClick = (e) => {
        const {isEnabled, limitMode, value: oldValue} = this.props;

        if (isEnabled) {
            const value = Math.min(mapInputToNumber(e.currentTarget.innerText) + Number(oldValue), MAX_LIMIT);

            metrics.sendPay(['Нажатие на предложенное значение лимита', value]);
            this.updateLimits({value, limitMode: limitMode === LIMIT_MODES.NOLIMIT ? LIMIT_MODES.DAY : limitMode});
        }
    };
    closeLimitModePopup = () => this.setState({isLimitModeOpened: false});
    openLimitModePopup = () => this.setState({isLimitModeOpened: true});
    closeCurrencyPopup = () => this.setState({isCurrencyOptionsOpened: false});
    toggleCurrencyPopup = () =>
        this.props.isEnabled && this.setState({isCurrencyOptionsOpened: !this.state.isCurrencyOptionsOpened});
    renderLimitOption = (value) => {
        const {limitMode} = this.props;

        return (
            <div
                className={b('option', {current: limitMode === value})}
                data-value={value}
                onClick={this.onLimitOptionClick}
                key={value}
            >
                {FRAME_OPTIONS_TEXTS[value]}
            </div>
        );
    };
    renderCurrencyOption = (value) => {
        const {payFormLimitCurrency} = this.props;

        return (
            <div
                className={b('option', {current: payFormLimitCurrency === value})}
                data-value={value}
                onClick={this.onCurrencyOptionClick}
                key={value}
            >
                {CURRENCIES_TEXTS[value]}
            </div>
        );
    };

    render() {
        const {limitMode, value, payFormLimitCurrency, isEnabled} = this.props;
        const {isLimitModeOpened, isCurrencyOptionsOpened} = this.state;

        const isLimitModePopupOpened = isEnabled && isLimitModeOpened;
        const isCurrencyPopupOpened = isEnabled && isCurrencyOptionsOpened;

        return (
            <div className={b({disabled: !isEnabled, enabled: isEnabled})} ref={this.mainRef}>
                <div className={b('top')}>
                    <div className={b('title')}>{i18n('Family.payLimitsInput.top.title')}</div>
                    <div className={b('frame')} onClick={this.openLimitModePopup}>
                        <span>{FRAME_OPTIONS_TEXTS[limitMode]}</span>
                        <div className={b('arrow', {rotate: isLimitModePopupOpened})} />
                    </div>
                    <Popup
                        hasTail={false}
                        directions={['bottom-center']}
                        target='anchor'
                        mainOffset={-187}
                        anchor={this.mainRef}
                        visible={isLimitModePopupOpened}
                        onClose={this.closeLimitModePopup}
                        zIndex={2500}
                        scope={this.mainRef}
                    >
                        <div className={b('mode')}>
                            <div className={b('title', {mode: true})}>{i18n('Family.payLimitsInput.top.title')}:</div>
                            <FamilyUniversalTileList
                                tiles={[
                                    {
                                        type: 'button',
                                        title: i18n('Family.payLimitsInput.modes.nolimit.title'),
                                        description: i18n('Family.payLimitsInput.modes.nolimit.description'),
                                        iconLeft: 'infinity',
                                        iconLeftBg: true,
                                        iconRight: limitMode === LIMIT_MODES.NOLIMIT && 'check',
                                        onClick: this.onLimitModeClickFuncs[LIMIT_MODES.NOLIMIT],
                                        dataT: 'pay-limits-input:limit-option:no-limit'
                                    },
                                    {
                                        type: 'button',
                                        title: i18n('Family.payLimitsInput.modes.total.title'),
                                        description: i18n('Family.payLimitsInput.modes.total.description'),
                                        iconLeft: 'download',
                                        iconLeftBg: true,
                                        iconRight: limitMode === LIMIT_MODES.TOTAL && 'check',
                                        onClick: this.onLimitModeClickFuncs[LIMIT_MODES.TOTAL],
                                        dataT: 'pay-limits-input:limit-option:total'
                                    }
                                ]}
                            />
                            <div className={b('text')}>{i18n('Family.payLimitsInput.modes.text')}</div>
                            <FamilyUniversalTileList
                                tiles={[
                                    {
                                        type: 'button',
                                        title: i18n('Family.payLimitsInput.modes.day.title'),
                                        iconLeft: 'repeat',
                                        iconLeftBg: true,
                                        iconRight: limitMode === LIMIT_MODES.DAY && 'check',
                                        onClick: this.onLimitModeClickFuncs[LIMIT_MODES.DAY],
                                        dataT: 'pay-limits-input:limit-option:day'
                                    },
                                    {
                                        type: 'button',
                                        title: i18n('Family.payLimitsInput.modes.week.title'),
                                        iconLeft: 'repeat',
                                        iconLeftBg: true,
                                        iconRight: limitMode === LIMIT_MODES.WEEK && 'check',
                                        onClick: this.onLimitModeClickFuncs[LIMIT_MODES.WEEK],
                                        dataT: 'pay-limits-input:limit-option:week'
                                    },
                                    {
                                        type: 'button',
                                        title: i18n('Family.payLimitsInput.modes.month.title'),
                                        iconLeft: 'repeat',
                                        iconLeftBg: true,
                                        iconRight: limitMode === LIMIT_MODES.MONTH && 'check',
                                        onClick: this.onLimitModeClickFuncs[LIMIT_MODES.MONTH],
                                        dataT: 'pay-limits-input:limit-option:month'
                                    }
                                ]}
                            />
                        </div>
                    </Popup>
                </div>
                <div className={b('inputWrap')}>
                    {limitMode === LIMIT_MODES.NOLIMIT ? (
                        <div className={b('inputNoLimit')}>{i18n('Family.payLimitsInput.nolimit')}</div>
                    ) : (
                        <>
                            <div className={b('input')}>
                                <Field
                                    size='l'
                                    view='big-input'
                                    disabled={!isEnabled}
                                    name='family-pay-limit-input'
                                    id='family-pay-limit-input'
                                    autoComplete='off'
                                    placeholder={i18n('Family.payLimitsInput.placeholder')}
                                    value={String(value)}
                                    onChange={this.onInputChange}
                                />
                            </div>
                            <div
                                className={b('currency')}
                                ref={this.currencyOptionsRef}
                                onClick={this.toggleCurrencyPopup}
                            >
                                {addCurrencySymbol(null, payFormLimitCurrency)}
                                <div className={b('arrow', {rotate: isCurrencyPopupOpened, currency: true})} />
                            </div>
                            <Popup
                                hasTail={false}
                                directions={['bottom-right']}
                                target='anchor'
                                mainOffset={0}
                                anchor={this.currencyOptionsRef}
                                visible={isCurrencyPopupOpened}
                                onClose={this.closeCurrencyPopup}
                                zIndex={2000}
                            >
                                <div className={b('options')}>{CURRENCIES_LIST.map(this.renderCurrencyOption)}</div>
                            </Popup>
                        </>
                    )}
                </div>
                <div className={b('suggest')}>
                    {SUGGEST_VALUES.map((value) => (
                        <div className={b('value')} key={value} onClick={this.onValueClick}>
                            {value}
                        </div>
                    ))}
                </div>
            </div>
        );
    }
}

FamilyPayLimitsInput.propTypes = {
    value: PropTypes.string,
    limitMode: PropTypes.string,
    placeId: PropTypes.string,
    currency: PropTypes.string,
    payFormLimitCurrency: PropTypes.string,
    currentSlot: PropTypes.object,
    isEnabled: PropTypes.bool,
    updatePayLimitForm: PropTypes.func,
    defaultValue: PropTypes.number
};
