import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Link} from '@components/Link';
import {PAY_SERVICES} from '../../../const';
import {metrics} from '../../../utils/metrics';
import './FamilyPayLimitsServices.styl';

const b = cn('FamilyPayLimitsServices');

const services = Object.keys(PAY_SERVICES).map((service) => ({name: PAY_SERVICES[service], id: service}));

export class FamilyPayLimitsServices extends React.Component {
    state = {height: 0, services};
    ref = React.createRef(null);
    componentDidMount() {
        this.filterServices();
    }
    filterServices = () => {
        const {hasPayTankerExp, hasPayAfishaExp, hasPayHealthExp, hasPayHelpExp} = this.props;

        if (!hasPayTankerExp || !hasPayAfishaExp || !hasPayHealthExp || !hasPayHelpExp) {
            this.setState({
                services: services.filter(
                    (service) =>
                        (service.name !== PAY_SERVICES.tanker || hasPayTankerExp) &&
                        (service.name !== PAY_SERVICES.afisha || hasPayAfishaExp) &&
                        (service.name !== PAY_SERVICES.health || hasPayHealthExp) &&
                        (service.name !== PAY_SERVICES.help || hasPayHelpExp)
                )
            });
        }
    };
    onServiceClick = (e) => {
        const {updatePayLimitForm, placeId, allowedServices = []} = this.props;
        const {id} = e.currentTarget.dataset;
        const isServiceAllowed = allowedServices.includes(id);

        metrics.sendPay(['Изменено состояние сервиса', id, isServiceAllowed ? 'Заблокирован' : 'Разрешен']);
        updatePayLimitForm({
            placeId,
            allowedServices: isServiceAllowed
                ? allowedServices.filter((service) => service !== id)
                : [...allowedServices, id]
        });
    };
    componentDidUpdate(props) {
        const {opened} = this.props;

        if (opened !== props.opened) {
            this.updateHeight();
        }
    }
    updateHeight = () => {
        const {opened = false, showDisclaimer = true} = this.props;

        const realHeight = this.ref.current.scrollHeight;
        const negativeMarginBottom = showDisclaimer ? 0 : 16;

        this.setState({height: opened ? realHeight - negativeMarginBottom : 0});
    };
    render() {
        const {allowedServices = [], readOnly = false, showDisclaimer = true, allowAllServices = false} = this.props;
        const {height, services} = this.state;

        return (
            <div className={b({clickable: !readOnly})} style={{height: `${height}px`}} ref={this.ref}>
                <div className={b('margin')} />
                {services.map(({name, id}) => {
                    const isAllowed = allowAllServices || allowedServices.includes(id);

                    return (
                        <div
                            className={b('service')}
                            key={id}
                            data-id={id}
                            onClick={readOnly ? undefined : this.onServiceClick}
                        >
                            <div className={b('icon', {blocked: !isAllowed, allowed: isAllowed, service: id})} />
                            <div className={b('name')}>{name}</div>
                        </div>
                    );
                })}
                {showDisclaimer && (
                    <div className={b('info')}>
                        {i18n('Family.payLimitsServices.info')}{' '}
                        <Link href='https://yandex.ru/support/id/family/pay.html' weight='medium' view='default'>
                            {i18n('Family.payLimitsServices.link')}
                        </Link>
                    </div>
                )}
            </div>
        );
    }
}

FamilyPayLimitsServices.propTypes = {
    opened: PropTypes.bool,
    readOnly: PropTypes.bool,
    showDisclaimer: PropTypes.bool,
    allowAllServices: PropTypes.bool,
    hasPayTankerExp: PropTypes.bool,
    hasPayAfishaExp: PropTypes.bool,
    hasPayHealthExp: PropTypes.bool,
    hasPayHelpExp: PropTypes.bool,
    placeId: PropTypes.string,
    allowedServices: PropTypes.arrayOf(PropTypes.string),
    updatePayLimitForm: PropTypes.func
};
