import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {getSlotType} from '../../utils/getSlotType';
import {getSlotSubtitle} from '../../utils/getSlotSubtitle';
import {PAGES} from '../../const';
import {FamilyUniversalTileList} from '../FamilyUniversalTileList';
import {FamilyTileLimitInfo} from '../FamilyTileLimitInfo';
import {FamilySlotsListCard} from './Card';

import './FamilySlotsList.styl';

const b = cn('FamilySlotsList');

const getTitle = ({isUser, isInvite, name}) => {
    if (isUser) {
        return (name || '').replace(/[-]/g, '‑');
    }
    return isInvite ? i18n('Family.slots-list.invite.title') : i18n('Family.slots-list.empty.title');
};

const getRenderSlots = (slots = [], hideFamilyInvite = false, isTouch = false, onSlotClick) => {
    const renderSlots = [];

    slots.forEach((slot, i) => {
        if (slot.isEmpty && hideFamilyInvite) {
            return;
        }

        const newSlot = {
            dataT: `slots-list:slot:${getSlotType(slot)}:${i}`,
            onClick: () => onSlotClick(slot),
            title: getTitle(slot),
            description: getSlotSubtitle(slot)
        };

        renderSlots.push(
            isTouch
                ? {
                      type: 'button',
                      iconRight: 'arrowRight',
                      iconRightPosition: 'top',
                      iconRightComponent: <FamilyTileLimitInfo slot={slot} />,
                      avatarLeft: slot,
                      avatarSize: 'xs',
                      titleEllipsisOverflow: true,
                      ...newSlot
                  }
                : {
                      slot,
                      children: <FamilyTileLimitInfo slot={slot} isRow={true} />,
                      ...newSlot
                  }
        );
    });

    return renderSlots;
};

export class FamilySlotsList extends React.Component {
    state = {renderSlots: []};
    constructor(props) {
        super(props);

        this.state = {
            renderSlots: getRenderSlots(props.slots, props.hideFamilyInvite, props.isTouch, this.onSlotClick)
        };
    }
    onSlotClick = (slot) => {
        const {loading, setCurrentSlot, goToPage, goToInvitePage} = this.props;

        if (!loading && !slot.isStub) {
            setCurrentSlot(slot);
            slot.isEmpty ? goToInvitePage() : goToPage(PAGES.slotInfo);
        }
    };
    componentDidUpdate(props) {
        const {slots} = this.props;

        if (slots !== props.slots) {
            this.updateRenderSlots();
        }
    }
    componentDidMount() {
        this.updateRenderSlots();
    }

    updateRenderSlots = () => {
        const {slots, hideFamilyInvite, isTouch} = this.props;

        this.setState({renderSlots: getRenderSlots(slots, hideFamilyInvite, isTouch, this.onSlotClick)});
    };
    render() {
        const {isTouch} = this.props;
        const {renderSlots = []} = this.state;

        return isTouch ? (
            <div className={b()} data-t='slots-list:tile'>
                <FamilyUniversalTileList tiles={renderSlots} />
            </div>
        ) : (
            <div className={b({isCards: true})} data-t='slots-list:cards'>
                {renderSlots.map((slot, i) => (
                    <FamilySlotsListCard key={i} {...slot} />
                ))}
            </div>
        );
    }
}

FamilySlotsList.propTypes = {
    slots: PropTypes.array,
    loading: PropTypes.bool,
    isTile: PropTypes.bool,
    setCurrentSlot: PropTypes.func,
    goToPage: PropTypes.func,
    goToInvitePage: PropTypes.func,
    hideFamilyInvite: PropTypes.bool,
    isTouch: PropTypes.bool
};
