import React from 'react';
import PropTypes from 'prop-types';
import {FamilyPageSelectorFull} from '../page-selectors/common/full';
import {FamilyPageSelectorModal} from '../page-selectors/common/modal';
import {FamilyNotification} from '../components/FamilyNotification';
import {metrics} from '../utils/metrics';
import {sendShownPageMetric} from '../utils/sendShownPageMetric';
import {PAGES, ROUTES} from '../const';

export class Family extends React.Component {
    componentDidMount() {
        const {
            fetchFamily,
            fetchInvite,
            setFamilyInviteId,
            goToMainPage,
            goToInvitePage,
            goToLoadingPage,
            goToKiddishPage,
            family: {familyId, pageInfo, isAdminAccess} = {},
            history = {},
            location: {pathname},
            inviteId
        } = this.props;

        if (inviteId) {
            fetchInvite(inviteId);
            setFamilyInviteId(inviteId);
        }

        if (familyId) {
            this.onFamilyInit();
            return fetchFamily();
        }

        if (!pageInfo) {
            if (ROUTES.main.includes(pathname) || (ROUTES.invite.includes(pathname) && !isAdminAccess)) {
                goToLoadingPage();
            } else if (ROUTES.invite.includes(pathname)) {
                goToInvitePage();
            }
        }

        fetchFamily().done(this.onFamilyInit);

        if (history.listen && typeof history.listen === 'function') {
            this.unlistenHistory = history.listen((state, action) => {
                if (action === 'POP') {
                    if (ROUTES.main.includes(state.pathname)) {
                        goToMainPage({withHistory: false});
                    } else if (ROUTES.invite.includes(state.pathname)) {
                        goToInvitePage({withHistory: false});
                    } else if (ROUTES.kiddish.includes(state.pathname)) {
                        goToKiddishPage({withHistory: false});
                    }
                }
            });
        }
    }
    componentDidUpdate(prevProps) {
        const {location: {pathname} = {}, goToMainPage, goToKiddishPage, goToInvitePage} = this.props;
        const {location: {pathname: prevPathname} = {}} = prevProps;

        if (prevPathname !== pathname) {
            if (ROUTES.main.includes(pathname) && !ROUTES.after3ds.includes(prevPathname)) {
                goToMainPage({withHistory: false});
            } else if (ROUTES.invite.includes(pathname)) {
                goToInvitePage({withHistory: false});
            } else if (ROUTES.kiddish.includes(pathname)) {
                goToKiddishPage({withHistory: false});
            }
        }
    }
    componentWillUnmount() {
        if (this.unlistenHistory) {
            this.unlistenHistory();
        }
    }
    onFamilyInit = () => {
        const {
            location: {pathname},
            common: {isFamilyEnable} = {},
            family: {isAdminAccess, hasFamilySubscription, hasFamily},
            goToPage,
            goToMainPage
        } = this.props;

        if (!isFamilyEnable && !hasFamilySubscription) {
            metrics.send(['Показ страницы семьи с заглушкой']);
            return goToPage(PAGES.stub, false);
        }

        sendShownPageMetric(pathname, hasFamily, hasFamilySubscription);

        if (ROUTES.main.includes(pathname) || (ROUTES.invite.includes(pathname) && !isAdminAccess && hasFamily)) {
            goToMainPage();
        }
    };
    render() {
        return (
            <>
                <FamilyPageSelectorFull />
                <FamilyPageSelectorModal />
                <FamilyNotification />
            </>
        );
    }
}

Family.propTypes = {
    family: PropTypes.object,
    common: PropTypes.object,
    location: PropTypes.object,
    history: PropTypes.object,
    settings: PropTypes.object,
    fetchFamily: PropTypes.func,
    goToPage: PropTypes.func,
    goToMainPage: PropTypes.func,
    goToLoadingPage: PropTypes.func,
    goToInvitePage: PropTypes.func,
    goToKiddishPage: PropTypes.func,
    fetchInvite: PropTypes.func,
    setFamilyInviteId: PropTypes.func,
    inviteId: PropTypes.string
};

export default Family;
