import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Popup} from '@components/Popup';
import {FamilyLayout} from '../../components/FamilyLayout';
import {FamilyButton} from '../../components/FamilyButton';
import {FamilyKiddishInput} from '../../components/FamilyKiddishInput';
import {FamilySlider} from '../../components/FamilySlider';
import {FamilyAvatar} from '../../components/FamilyAvatar';
import {metrics} from '../../utils/metrics';
import {getKiddishMetricParams} from '../../utils/getKiddishMetricParams';
import {KIDDISH_PAGES_INFO, KIDDISH_PAGES_INFO_EDIT, PAGES} from '../../const';
import './FamilyInviteKiddishLayout.styl';

const b = cn('FamilyInviteKiddishLayout');

export class FamilyInviteKiddishLayout extends React.Component {
    ref = React.createRef();

    getPageInfo = () => {
        const {curPage, isEditing = false} = this.props;

        return (isEditing ? KIDDISH_PAGES_INFO_EDIT : KIDDISH_PAGES_INFO)[curPage] || {};
    };
    sendMetric = (params) => {
        const {isEditing = false} = this.props;
        const {type} = this.getPageInfo();

        metrics.sendKiddish(getKiddishMetricParams(isEditing, type, params));
    };

    onAvatarClick = () => this.props.goToPage(PAGES.kiddishAvatar);
    componentDidMount() {
        this.sendMetric('Открытие');
    }
    goBack = () => {
        const {goToKiddishPage} = this.props;

        this.sendMetric('Стрелка Назад');

        goToKiddishPage();
    };
    onPrevStepClick = () => {
        const {goToPage, onPrevStepClick} = this.props;

        goToPage(this.getPageInfo().prev);
        onPrevStepClick && onPrevStepClick();
    };
    onKiddishNextStepClick = (page) => {
        const {
            hasFamily = false,
            isEditing = false,
            isChildish = false,
            isKiddishOnlyThisStep = false,
            curPage,
            postKiddish,
            postEditKiddish,
            fetchFamily,
            onNextStepClick,
            goToPage,
            goToKiddishPage
        } = this.props;

        onNextStepClick && onNextStepClick();

        if (isChildish) {
            return;
        }

        const kiddishPagesInfo = isEditing ? KIDDISH_PAGES_INFO_EDIT : KIDDISH_PAGES_INFO;

        if (kiddishPagesInfo[page]) {
            return goToPage(page);
        }

        const currentKiddishPage = kiddishPagesInfo[curPage];

        if (isKiddishOnlyThisStep || (currentKiddishPage && !currentKiddishPage.next)) {
            return (isEditing ? postEditKiddish : postKiddish)()
                .done(() => {
                    metrics.sendKiddish(getKiddishMetricParams(isEditing, null, 'Успех'));

                    if (hasFamily) {
                        goToKiddishPage();
                        fetchFamily();
                    } else {
                        fetchFamily().done(() => goToKiddishPage());
                    }
                })
                .fail(() => metrics.sendKiddish(getKiddishMetricParams(isEditing, null, 'Ошибка')));
        }

        if (currentKiddishPage && currentKiddishPage.next) {
            goToPage(currentKiddishPage.next);
        }
    };

    render() {
        const {
            title,
            curPage,
            slider,
            inputs = [],
            isEditing = false,
            isChildish = false,
            isKiddishOnlyThisStep = false,
            isButtonDisabled = false,
            showAvatarSelector = false,
            isTouch
        } = this.props;

        const kiddishPagesInfo = isEditing ? KIDDISH_PAGES_INFO_EDIT : KIDDISH_PAGES_INFO;
        const {step, type, prev, next} = kiddishPagesInfo[curPage] || {};
        const isFinalStep = !next || isKiddishOnlyThisStep;
        const showBackButton = prev && !isKiddishOnlyThisStep;

        return (
            <FamilyLayout onBackButtonClick={this.goBack}>
                <div className={b({isTouch})} data-t='family:layout:invite-kiddish'>
                    <h3 className={b('process')}>
                        {isChildish
                            ? i18n('Family.kiddish-layout.process.childish')
                            : isEditing
                            ? i18n('Family.kiddish-layout.process.edit')
                            : i18n('Family.kiddish-layout.process.create')}
                    </h3>
                    {Boolean(step) && !isKiddishOnlyThisStep && (
                        <div className={b('step')}>
                            {i18n('Family.kiddish-layout.steps')
                                .replace('%1', step)
                                .replace('%2', kiddishPagesInfo.STEPS_COUNT)}
                        </div>
                    )}
                    <h2 className={b('title')}>{title}</h2>

                    <div className={b('content')}>
                        {showAvatarSelector && (
                            <>
                                <div className={b('avatar')} onClick={this.onAvatarClick} ref={this.ref}>
                                    <FamilyAvatar isCurrentSlot={true} size='l' isCentered={true} />
                                    {!isTouch && <div className={b('avatarChange')}>{i18n('_AUTH_.change')}</div>}
                                </div>
                                {isTouch && (
                                    <Popup
                                        hasTail={true}
                                        directions={['right-center', 'bottom-center', 'left-center', 'right-center']}
                                        target='anchor'
                                        mainOffset={8}
                                        anchor={this.ref}
                                        ignoreRefs={[this.ref]}
                                        visible={isTouch}
                                        view='black'
                                    >
                                        <div className={b('popup')}>{i18n('_AUTH_.change')}</div>
                                    </Popup>
                                )}
                            </>
                        )}
                        {Boolean(inputs && inputs.length) &&
                            inputs.map((input) => <FamilyKiddishInput {...input} key={input.field} />)}
                        {Boolean(slider) && <FamilySlider {...slider} />}
                    </div>

                    <div className={b('next')}>
                        <FamilyButton
                            view='action'
                            onClick={this.onKiddishNextStepClick}
                            disabled={isButtonDisabled}
                            text={
                                isFinalStep && !isChildish
                                    ? isEditing
                                        ? i18n('Frontend.save')
                                        : i18n('Family.kiddish-layout.create')
                                    : i18n('Profile.common.continue')
                            }
                            metricKiddish={getKiddishMetricParams(isEditing, type, 'Продолжить')}
                        />
                    </div>

                    {showBackButton && (
                        <div className={b('back')}>
                            <FamilyButton
                                view='default'
                                onClick={this.onPrevStepClick}
                                text={i18n('_AUTH_.common.back')}
                                metricKiddish={getKiddishMetricParams(isEditing, type, 'Назад')}
                            />
                        </div>
                    )}
                </div>
            </FamilyLayout>
        );
    }
}

FamilyInviteKiddishLayout.propTypes = {
    title: PropTypes.string,
    curPage: PropTypes.string,
    slider: PropTypes.object,
    inputs: PropTypes.array,
    isEditing: PropTypes.bool,
    isKiddishOnlyThisStep: PropTypes.bool,
    showAvatarSelector: PropTypes.bool,
    hasFamily: PropTypes.bool,
    isTouch: PropTypes.bool,
    isChildish: PropTypes.bool,
    isButtonDisabled: PropTypes.bool,
    hideBackButton: PropTypes.bool,
    onPrevStepClick: PropTypes.func,
    onNextStepClick: PropTypes.func,
    updateKiddish: PropTypes.func,
    goToPage: PropTypes.func,
    goToMainPage: PropTypes.func,
    postKiddish: PropTypes.func,
    postEditKiddish: PropTypes.func,
    fetchFamily: PropTypes.func,
    goToKiddishPage: PropTypes.func
};
