import React from '@plibs/react';
import PropTypes from 'prop-types';
import {FamilyActionLayout} from '@blocks/morda/new/family/layouts/FamilyActionLayout';
import {redirectToRetpathFromWebview} from '@blocks/utils';
import {WEBVIEW_STATUS_CLOSE} from '@blocks/constants';
import {PAGES, ACCEPT_INVITE_TEXTS} from '@blocks/morda/new/family/const';

export class FamilyAcceptInvitePage extends React.Component {
    state = {
        isLoading: false
    };

    componentDidUpdate(prevProps) {
        const {isInviteConfirmed, isWebview, retpath, goToPage} = this.props;

        if (!prevProps.isInviteConfirmed && isInviteConfirmed) {
            if (isWebview && retpath) {
                const canBeModal = false;

                return goToPage(PAGES.inviteWelcome, canBeModal);
            }

            location.href = '/profile/family/welcome';
        }
    }

    removeFamily = () => {
        const {removeFamily} = this.props;

        removeFamily().done((result = {}) => {
            if (result.status === 'ok') {
                this.acceptInvite();
            }
        });
    };

    handleInviteClick = () => {
        const {hasKidsProfiles} = this.props;

        this.setState = {
            isLoading: true
        };

        if (hasKidsProfiles) {
            return this.removeFamily();
        }

        return this.acceptInvite();
    };

    acceptInvite = () => {
        const {confirmInvite, inviteId} = this.props;

        if (inviteId) {
            confirmInvite(inviteId);
        }
    };

    cancelInvite = () => {
        const {goToMainPage, retpath, isWebview} = this.props;

        if (isWebview) {
            return retpath
                ? redirectToRetpathFromWebview(retpath, WEBVIEW_STATUS_CLOSE)
                : (location.href = '/profile/family');
        }

        goToMainPage();
    };

    getTexts = () => {
        const {hasFamily, hasMembers, hasKidsProfiles, isInviteInYourFamily, isAdmin, inviter} = this.props;
        const {name} = inviter;

        if (isInviteInYourFamily) {
            return {
                title: ACCEPT_INVITE_TEXTS.titles.yourFamily,
                description: isAdmin
                    ? ACCEPT_INVITE_TEXTS.texts.yourFamily.admin
                    : ACCEPT_INVITE_TEXTS.texts.yourFamily.member
            };
        }

        if (isAdmin) {
            if (hasMembers) {
                return {
                    title: ACCEPT_INVITE_TEXTS.titles.inviteWarnAdminWithMembers,
                    description: ACCEPT_INVITE_TEXTS.texts.inviteWarnAdminWithMembers
                };
            }

            if (hasKidsProfiles) {
                return {
                    title: ACCEPT_INVITE_TEXTS.titles.invite.replace('%name', name),
                    description: ACCEPT_INVITE_TEXTS.texts.inviteKidsWarn
                };
            }
        }

        if (hasFamily) {
            return {
                title: ACCEPT_INVITE_TEXTS.titles.inviteWarn.replace('%name', name),
                description: ACCEPT_INVITE_TEXTS.texts.inviteWarnUser
            };
        }

        return {
            title: ACCEPT_INVITE_TEXTS.titles.invite.replace('%name', name),
            description: ACCEPT_INVITE_TEXTS.texts.invite
        };
    };

    getButtons = () => {
        const {
            isInviteInYourFamily,
            isAdmin,
            hasMembers,
            hasKidsProfiles,
            hasFamily,
            hasProcessingInvitations
        } = this.props;

        if (isInviteInYourFamily) {
            return [
                {
                    text: ACCEPT_INVITE_TEXTS.buttons.ok,
                    view: 'action',
                    metric: ['Приглашение недействительно: нажали на кнопку "Понятно"'],
                    onClick: this.cancelInvite
                }
            ];
        }

        if (isAdmin) {
            if (hasMembers || (hasProcessingInvitations && !hasKidsProfiles)) {
                return [
                    {
                        text: ACCEPT_INVITE_TEXTS.buttons.ok,
                        view: 'action',
                        metric: ['Админ семьи, принятие приглашения: нажали на кнопку "Понятно"'],
                        onClick: this.cancelInvite
                    },
                    {
                        text: ACCEPT_INVITE_TEXTS.buttons.changeFamily,
                        view: 'transparent',
                        metric: ['Админ семьи, принятие приглашения: нажали на кнопку "Перейти в новую семью"'],
                        onClick: this.showWarning
                    }
                ];
            }

            if (hasKidsProfiles) {
                return [
                    {
                        text: ACCEPT_INVITE_TEXTS.buttons.joinFamily,
                        view: 'action',
                        progress: this.state.isLoading,
                        metric: ['Админ семьи, принятие приглашения: нажали на кнопку "Вступить в семью"'],
                        onClick: this.handleInviteClick
                    },
                    {
                        text: ACCEPT_INVITE_TEXTS.buttons.doNothing,
                        view: 'transparent',
                        metric: ['Админ семьи, принятие приглашения: нажали на кнопку "Оставить все как есть"'],
                        onClick: this.cancelInvite
                    }
                ];
            }

            return [
                {
                    text: ACCEPT_INVITE_TEXTS.buttons.joinFamily,
                    view: 'action',
                    progress: this.state.isLoading,
                    metric: ['Принятие семейного приглашения: нажали на кнопку "Отлично"'],
                    onClick: this.removeFamily
                }
            ];
        }

        if (hasFamily) {
            return [
                {
                    text: ACCEPT_INVITE_TEXTS.buttons.stayInFamily,
                    view: 'action',
                    metric: ['Участник семьи, принятие приглашения: нажали на кнопку "Остаться в текущей семье"'],
                    onClick: this.cancelInvite
                },
                {
                    text: ACCEPT_INVITE_TEXTS.buttons.changeFamily,
                    view: 'transparent',
                    metric: ['Участник семьи, принятие приглашения: нажали на кнопку "Перейти в новую семью"'],
                    onClick: this.showWarning
                }
            ];
        }

        return [
            {
                text: ACCEPT_INVITE_TEXTS.buttons.joinFamily,
                view: 'action',
                progress: this.state.isLoading,
                metric: ['Принятие семейного приглашения: нажали на кнопку "Отлично"'],
                onClick: this.acceptInvite
            }
        ];
    };

    showWarning = () => {
        this.props.goToPage(PAGES.inviteAcceptWarning, true, true, false, PAGES.inviteAccept);
    };

    render() {
        const {inviter, isInviteInYourFamily} = this.props;
        const {title, description} = this.getTexts();

        return (
            <FamilyActionLayout
                hideBackButton={true}
                decor={isInviteInYourFamily ? 'bubble' : undefined}
                avatar={isInviteInYourFamily ? undefined : inviter}
                title={title}
                text={description}
                buttons={this.getButtons()}
                dataTPostfix={'invite'}
            />
        );
    }
}

FamilyAcceptInvitePage.propTypes = {
    hasFamily: PropTypes.bool,
    hasKidsProfiles: PropTypes.bool,
    hasProcessingInvitations: PropTypes.bool,
    hasMembers: PropTypes.bool,
    loading: PropTypes.bool,
    isInviteConfirmed: PropTypes.bool,
    isAdmin: PropTypes.bool,
    goToMainPage: PropTypes.func,
    goToPage: PropTypes.func,
    removeFamily: PropTypes.func,
    confirmInvite: PropTypes.func,
    inviteId: PropTypes.string,
    inviter: PropTypes.object,
    isWebview: PropTypes.bool,
    isInviteInYourFamily: PropTypes.bool,
    retpath: PropTypes.string
};
