import React from '@plibs/react';
import PropTypes from 'prop-types';
import {FamilyActionLayout} from '@blocks/morda/new/family/layouts/FamilyActionLayout';
import {Field} from '@components/Field';
import {metrics} from '../../utils/metrics';
import {ROUTES} from '../../const';

export const INVITE_FORM_TYPES = ['smsOrMail', 'link'];
const INVITE_INPUT_ID = 'family_invite_input';
const dataTPrefix = 'family:pages:invite-redesign';

export class FamilyRedesignInvitePage extends React.Component {
    fieldRef = React.createRef();

    componentDidMount() {
        if (this.fieldRef.current && this.fieldRef.current.focus) {
            this.fieldRef.current.focus();
        }
    }

    componentWillUnmount() {
        const {setFamilyInviteTarget, defaultInviteTarget, replaceHistory} = this.props;

        this.clearInput();
        setFamilyInviteTarget(defaultInviteTarget);
        replaceHistory(ROUTES.main[0]);
    }

    clearInput = () => {
        this.props.updateValues({
            field: 'family_invite_input',
            value: ''
        });
    };

    completeAction = () => {
        const {goToMainPage, notRedirectAfterInvite} = this.props;

        !notRedirectAfterInvite && goToMainPage();
    };

    onInviteCreated = async (contact) => {
        const {fetchFamily, setFamilyInviteContact, toggleFamilyNotification} = this.props;

        metrics.goalPay(contact ? 'add_email_phone' : 'invitation_sent');

        if (contact) {
            setFamilyInviteContact(contact);
            this.completeAction();

            toggleFamilyNotification({
                isVisible: true,
                iconType: 'ok',
                text: i18n('Family.member.name.invited')
            });
        } else {
            await this.copyInviteLink();
        }

        fetchFamily();
    };

    canPostInvite = () => {
        const {slots = [], loading} = this.props;

        return !loading && (!slots.length || slots.some((slot) => slot.isEmpty));
    };

    sendInvite = (event) => {
        const {inviteInputValue, postInvite} = this.props;

        if (event) {
            event.preventDefault();
        }

        if (inviteInputValue && this.canPostInvite()) {
            postInvite(inviteInputValue).done(() => this.onInviteCreated(inviteInputValue));
        }
    };

    shareInvite = () => {
        const {postInvite} = this.props;

        if (this.canPostInvite()) {
            postInvite().done(() => this.onInviteCreated());
        }
    };

    toggleInviteTarget = () => {
        const {setFamilyInviteTarget} = this.props;

        setFamilyInviteTarget(INVITE_FORM_TYPES[0]);
    };

    copyInviteLink = async () => {
        const {inviteLink, toggleFamilyNotification} = this.props;
        const {share, clipboard: {writeText} = {}} = navigator;
        const shareIsAvailable = share && typeof share === 'function';
        const copyIsAvailable = writeText && typeof writeText === 'function';

        if (inviteLink) {
            if (shareIsAvailable) {
                try {
                    await navigator.share({title: i18n('Family.invite.share_link_title'), url: inviteLink});

                    return this.completeAction();
                } catch (e) {
                    console.error('error throwed when share invite link:', e);
                }
            }

            if (copyIsAvailable) {
                try {
                    await navigator.clipboard.writeText(inviteLink);

                    toggleFamilyNotification({
                        isVisible: true,
                        iconType: 'ok',
                        text: i18n('Family.invite-copy.message')
                    });

                    return this.completeAction();
                } catch (e) {
                    console.error('error throwed when copy invite link:', e);
                }
            }

            this.completeAction();

            toggleFamilyNotification({
                theme: 'link',
                isVisible: true,
                text: i18n('Family.invite-copy.message-done').replace(
                    '%link',
                    `<span class="inviteLink">${inviteLink}</span>`
                )
            });
        }
    };

    render() {
        const {inviteTarget, hideBackButton, inviteInputValue} = this.props;

        return (
            <div>
                {inviteTarget === INVITE_FORM_TYPES[1] ? (
                    <FamilyActionLayout
                        hideBackButton={hideBackButton}
                        decor='invite'
                        title={i18n('Family.invite.share-title')}
                        text={i18n('Family.invite.share-text')}
                        buttons={[
                            {
                                text: i18n('Family.invite.share-link'),
                                icon: 'share',
                                view: 'action',
                                metric: ['Нажатие на кнопку "Сгенерировать приглашение"'],
                                dataT: `${dataTPrefix}:link:buttons:share`,
                                onClick: this.shareInvite
                            },
                            {
                                text: i18n('Family.invite.send-invite'),
                                view: 'transparent',
                                metric: ['Нажатие на кнопку "Отправить смс или письмо"'],
                                dataT: `${dataTPrefix}:link:buttons:invite`,
                                onClick: this.toggleInviteTarget
                            }
                        ]}
                        dataTPostfix={`redesign-invite-${INVITE_FORM_TYPES[1]}`}
                    />
                ) : (
                    <form method='POST' onSubmit={this.sendInvite}>
                        <FamilyActionLayout
                            hideBackButton={hideBackButton}
                            decor='invite'
                            title={i18n('Family.invite.send-invitation-title')}
                            text={i18n('Family.invite.send-invitation-text')}
                            buttons={[
                                {
                                    text: i18n('Family.invite.send-btn'),
                                    icon: 'send',
                                    view: 'action',
                                    metric: ['Отправка приглашения через инпут'],
                                    dataT: `${dataTPrefix}:contact:buttons:invite`,
                                    onClick: this.sendInvite,
                                    disabled: !inviteInputValue
                                },
                                {
                                    text: i18n('Family.invite.share-link'),
                                    icon: 'share',
                                    view: 'transparent',
                                    metric: ['Нажатие на кнопку "Сгенерировать приглашение"'],
                                    dataT: `${dataTPrefix}:contact:buttons:share`,
                                    onClick: this.shareInvite
                                }
                            ]}
                            dataTPostfix={'redesign-invite'}
                        >
                            <Field
                                ref={this.fieldRef}
                                size='l'
                                placeholder={i18n('Family.invite-input.placeholder')}
                                name={INVITE_INPUT_ID}
                                dataT={`${dataTPrefix}:contact:input`}
                                type='text'
                                view='big-input'
                            />
                        </FamilyActionLayout>
                    </form>
                )}
            </div>
        );
    }
}

FamilyRedesignInvitePage.propTypes = {
    inviteInputValue: PropTypes.string,
    inviteTarget: PropTypes.string.isRequired,
    defaultInviteTarget: PropTypes.string,
    inviteLink: PropTypes.string,
    loading: PropTypes.bool,
    hideBackButton: PropTypes.bool,
    notRedirectAfterInvite: PropTypes.bool,
    slots: PropTypes.array,
    fetchFamily: PropTypes.func,
    setFamilyInviteContact: PropTypes.func,
    postInvite: PropTypes.func,
    setFamilyInviteTarget: PropTypes.func.isRequired,
    goToMainPage: PropTypes.func,
    toggleFamilyNotification: PropTypes.func.isRequired,
    updateValues: PropTypes.func.isRequired,
    replaceHistory: PropTypes.func.isRequired
};
