import React from 'react';
import PropTypes from 'prop-types';
import {redirectToRetpathFromWebview} from '@blocks/utils';
import {WEBVIEW_STATUS_CLOSE} from '@blocks/constants';
import {FamilySlotInfoLayout} from '../../layouts/FamilySlotInfoLayout';
import {FamilyPaySettingsBanner} from '../../components/FamilyPaySettings/Banner';
import {FamilyWarning} from '../../components/FamilyWarning';
import {FamilyKiddishInfo} from '../../components/FamilyKiddishInfo';
import {addCurrencySymbol} from '../../utils/currency';
import {shortenLongNames} from '../../utils/shortenLongNames';
import {shortenInviteContact} from '../../utils/shortenInviteContact';
import {getSlotSubtitle} from '../../utils/getSlotSubtitle';
import {LIMIT_MODES, PAGES, ROUTES} from '../../const';

const dataTPrefix = 'family:pages:slot-info';

export class FamilySlotInfoPage extends React.Component {
    onCancelInviteClick = () => this.props.goToPage(PAGES.inviteCancel);
    onExcludeClick = () => this.props.goToPage(PAGES.memberExclude);
    onLeaveClick = () => this.props.goToPage(PAGES.memberLeave);
    onRemoveKiddishClick = () => this.props.goToPage(PAGES.kiddishRemove);
    onControlChildishClick = () => this.props.goToPage(PAGES.childishControl);
    onOrderHistoryClick = () => {
        ROUTES.orderHistory[0];
    };
    onEditKiddishClick = () => {
        const {updateCurrentSlot, goToPage} = this.props;

        updateCurrentSlot({isKiddishOnlyThisStep: false});
        goToPage(PAGES.kiddishName);
    };
    onLimitsSettingClick = () => {
        const {goToPage, setCurrentSlot, currentSlot} = this.props;

        setCurrentSlot({...currentSlot, isPrefilledForLimitPage: true});
        goToPage(PAGES.payLimits);
    };
    onSetNoLimitClick = () => {
        const {
            currentSlot,
            loading,
            updatePayLimitForm,
            postUsersLimits,
            resetPayLimitsForm,
            fetchFamily,
            setCurrentSlot,
            limitValue
        } = this.props;

        if (!loading) {
            updatePayLimitForm({
                placeId: currentSlot.placeId,
                limit: {limitMode: LIMIT_MODES.NOLIMIT, value: limitValue}
            });
            postUsersLimits().done(() => {
                resetPayLimitsForm();
                fetchFamily().done(() => setCurrentSlot(currentSlot));
            });
        }
    };
    onFillBalanceClick = () => {
        const {
            currentSlot,
            limitValue,
            balance,
            postFillBalance,
            loading,
            setCurrentSlot,
            fetchFamily,
            toggleFamilyNotification
        } = this.props;

        if (balance === limitValue) {
            return toggleFamilyNotification({
                isVisible: true,
                iconType: 'ok',
                text: i18n('Family.pages.slot-info.notification.balance')
            });
        }

        if (!loading) {
            postFillBalance(currentSlot.placeId).done(() => fetchFamily().done(() => setCurrentSlot(currentSlot)));
        }
    };
    goBack = () => {
        const {goToMainPage, goToKiddishPage, currentSlot, isWebview, retpath} = this.props;

        if (isWebview && retpath) {
            return redirectToRetpathFromWebview(retpath, WEBVIEW_STATUS_CLOSE);
        }

        currentSlot.isKiddish ? goToKiddishPage() : goToMainPage();
    };
    getTitle = () => {
        const {currentSlot: {isUser, isKiddish, isInvite, name, inviteContact} = {}} = this.props;

        if (isUser || isKiddish) {
            return shortenLongNames(name);
        }

        if (isInvite) {
            return inviteContact ? (
                <>
                    {i18n('Family.pages.slot-info.title.invite')}
                    <br />
                    {shortenInviteContact(inviteContact)}
                </>
            ) : (
                i18n('Family.pages.slot-info.title.invite')
            );
        }
    };
    getDescription = () => {
        const {currentSlot: {isInvite} = {}} = this.props;

        if (isInvite) {
            return i18n('Family.pages.slot-info.description.invite');
        }
    };
    getTiles = () => {
        const {
            currentSlot,
            passportLinks,
            currency,
            limitMode,
            limitValue,
            isAdminAccess,
            isCardActualyEnabled,
            language
        } = this.props;
        const showPayBlock = this.isShowPayBlock();

        if (currentSlot.isFake) {
            return [];
        }

        if (currentSlot.isYou) {
            return [
                {
                    type: 'button',
                    title: isAdminAccess
                        ? i18n('Family.pages.slot-info.tiles.your.leave.title.admin')
                        : i18n('Family.pages.slot-info.tiles.your.leave.title.member'),
                    iconLeft: 'ban',
                    iconLeftBg: true,
                    dataT: `${dataTPrefix}:controls:you:leave`,
                    onClick: this.onLeaveClick
                }
            ];
        }

        if (currentSlot.isKiddish) {
            return [
                {
                    type: 'button',
                    title: i18n('Frontend.common.edit'),
                    iconLeft: 'edit',
                    iconLeftBg: true,
                    dataT: `${dataTPrefix}:controls:kiddish:edit`,
                    onClick: this.onEditKiddishClick
                },
                {
                    type: 'button',
                    title: i18n('Family.pages.slot-info.tiles.kiddish.remove.title'),
                    iconLeft: 'ban',
                    iconLeftBg: true,
                    dataT: `${dataTPrefix}:controls:kiddish:remove`,
                    onClick: this.onRemoveKiddishClick
                }
            ];
        }

        if (currentSlot.isUser && isAdminAccess) {
            const increaseBalanceTile = {
                type: 'button',
                title: i18n('Family.pages.slot-info.tiles.user.balance.title').replace(
                    '%s',
                    addCurrencySymbol(limitValue, currency, language)
                ),
                description: i18n('Family.pages.slot-info.tiles.user.balance.description'),
                iconLeft: '4plus',
                iconLeftBg: true,
                dataT: `${dataTPrefix}:controls:user:balance`,
                onClick: this.onFillBalanceClick
            };
            const setNoLimitTile = {
                type: 'button',
                title: i18n('Family.pages.slot-info.tiles.user.nolimit.title'),
                description: i18n('Family.pages.slot-info.tiles.user.nolimit.description'),
                iconLeft: 'infinity',
                iconLeftBg: true,
                dataT: `${dataTPrefix}:controls:user:nolimit`,
                onClick: this.onSetNoLimitClick
            };
            const limitsSettingsTile = {
                type: 'button',
                title: i18n('Family.pages.slot-info.tiles.user.limit-settings.title'),
                iconLeft: 'download',
                iconLeftBg: true,
                iconRight: 'arrowRight',
                dataT: `${dataTPrefix}:controls:user:limit-settings`,
                onClick: this.onLimitsSettingClick
            };
            const excludeTile = {
                type: 'button',
                title: i18n('Family.pages.slot-info.tiles.user.exclude.title'),
                iconLeft: 'ban',
                iconLeftBg: true,
                dataT: `${dataTPrefix}:controls:user:exclude`,
                onClick: this.onExcludeClick
            };

            const payTiles =
                showPayBlock && currentSlot.isAbleToUsePay
                    ? limitMode !== LIMIT_MODES.NOLIMIT && isCardActualyEnabled
                        ? [increaseBalanceTile, setNoLimitTile, limitsSettingsTile]
                        : [limitsSettingsTile]
                    : [];

            if (currentSlot.isChildish) {
                const orderHistoryTile = {
                    type: 'link',
                    title: i18n('Family.pages.slot-info.tiles.child.order-history.title'),
                    iconLeft: 'rub',
                    iconLeftBg: true,
                    dataT: `${dataTPrefix}:controls:childish:order-history`,
                    href: passportLinks.orderHistory
                    // Нужна "фильтрация по детскому аккаунту"
                };
                const journalTile = {
                    type: 'link',
                    title: i18n('Family.pages.slot-info.tiles.child.journal.title'),
                    iconLeft: 'clock',
                    iconLeftBg: true,
                    dataT: `${dataTPrefix}:controls:childish:journal`,
                    href: `${passportLinks.journal}?origin=passport_family`
                    // Нужна "фильтрация по детскому аккаунту"
                };
                const controlChildishTile = {
                    type: 'button',
                    title: i18n('Family.member.actions.control'),
                    iconLeft: 'gear',
                    iconLeftBg: true,
                    dataT: `${dataTPrefix}:controls:childish:control`,
                    onClick: this.onControlChildishClick
                };

                return [...payTiles, orderHistoryTile, journalTile, controlChildishTile];
            }

            return [...payTiles, excludeTile];
        }

        return [];
    };
    getButtons = () => {
        const {currentSlot: {isInvite = false} = {}} = this.props;
        const closeButton = {
            text: i18n('Frontend.close'),
            view: 'default',
            dataT: `${dataTPrefix}:buttons:close`,
            onClick: this.goBack
        };
        const cancelInviteButton = {
            text: i18n('Family.pages.slot-info.buttons.cancel-invite'),
            view: 'transparent',
            dataT: `${dataTPrefix}:buttons:cancel-invite`,
            onClick: this.onCancelInviteClick
        };

        return [closeButton, isInvite && cancelInviteButton];
    };
    isShowPayBlock = () => {
        const {currentSlot, yourSlot, isAdminAccess, bound} = this.props;

        return isAdminAccess && currentSlot.isUser && !currentSlot.isAdmin && yourSlot.isAbleToUsePay && bound;
    };
    render() {
        const {currentSlot, hideBackButton} = this.props;

        const showPayBlock = this.isShowPayBlock();

        return (
            <FamilySlotInfoLayout
                title={this.getTitle()}
                subtitle={getSlotSubtitle(currentSlot)}
                description={this.getDescription()}
                hideBackButton={hideBackButton}
                onBackButtonClick={this.goBack}
                tiles={this.getTiles()}
                buttons={this.getButtons()}
                isBigMargins={Boolean(currentSlot.isInvite)}
            >
                {showPayBlock &&
                    (currentSlot.isAbleToUsePay ? (
                        <FamilyPaySettingsBanner slot={currentSlot} />
                    ) : (
                        <FamilyWarning
                            text={i18n('Family.payLimits.warning').replace('%name', shortenLongNames(currentSlot.name))}
                        />
                    ))}
                {currentSlot.isKiddish && <FamilyKiddishInfo kiddish={currentSlot} />}
            </FamilySlotInfoLayout>
        );
    }
}

FamilySlotInfoPage.propTypes = {
    currentSlot: PropTypes.object,
    yourSlot: PropTypes.object,
    passportLinks: PropTypes.object,
    isAdminAccess: PropTypes.bool,
    isCardActualyEnabled: PropTypes.bool,
    bound: PropTypes.bool,
    loading: PropTypes.bool,
    isWebview: PropTypes.bool,
    hideBackButton: PropTypes.bool,
    limitMode: PropTypes.string,
    currency: PropTypes.string,
    retpath: PropTypes.string,
    language: PropTypes.string,
    limitValue: PropTypes.number,
    balance: PropTypes.number,
    toggleFamilyNotification: PropTypes.func,
    updatePayLimitForm: PropTypes.func,
    updateCurrentSlot: PropTypes.func,
    postUsersLimits: PropTypes.func,
    postFillBalance: PropTypes.func,
    resetPayLimitsForm: PropTypes.func,
    fetchFamily: PropTypes.func,
    setCurrentSlot: PropTypes.func,
    goToPage: PropTypes.func,
    goToMainPage: PropTypes.func,
    goToKiddishPage: PropTypes.func
};
