import {findDOMNode} from 'react-dom';
import api from '@blocks/api';
import metrics from '@blocks/metrics';
import reloadCaptcha from '@components/Captcha/actions/reloadCaptcha';
import {push} from 'react-router-redux';
import {setAppPasswordsNumber} from '@blocks/morda/app_passwords/actions';

/* eslint-disable no-invalid-this */

const fields = ['currentPassword', 'newPassword', 'repeatPassword', 'captcha'];
const metricsSent = {};
const fieldsMetrics = {
    currentPassword: 'Заполнение текущего пароля',
    newPassword: 'Заполнение нового пароля',
    repeatPassword: 'Заполнение повторного пароля'
};
const METRICS_HEADER = 'Смена пароля';

export function sendOpenMetrics() {
    metrics.send([METRICS_HEADER, 'Открытие']);
}

export function sendToggleMetrics() {
    metrics.send([METRICS_HEADER, 'Нажатие на "изменить доступы"']);
}

export function getError(name, value, newPassword = this.state.newPassword) {
    if (value.length === 0) {
        return 'empty';
    }

    if (name === 'repeatPassword' && value !== newPassword) {
        return 'not_matched';
    }

    return false;
}

export function getErrors(name, value, newPassword, _errors = this.state.errors) {
    const errors = Object.assign({}, _errors);
    const error = getError.call(this, name, value, newPassword);

    if (!error && !errors.hasOwnProperty(name)) {
        return _errors;
    }

    if (error) {
        if (errors.hasOwnProperty(name)) {
            return _errors;
        }

        errors[name] = error;
    } else {
        delete errors[name];
    }

    return errors;
}

export function changeText({name, text}) {
    let errors = getErrors.call(this, name, text);

    if (name === 'repeatPassword' || name === 'newPassword') {
        if ((name === 'repeatPassword' ? text : this.state.repeatPassword).length) {
            if (name === 'repeatPassword') {
                errors = getErrors.call(this, 'repeatPassword', text, undefined, errors);
            } else {
                errors = getErrors.call(this, 'repeatPassword', this.state.repeatPassword, text, errors);
            }
        }
    }

    this.setState({
        [name]: text,
        errors
    });
}

export function checkForErrors() {
    return fields.reduce((acc, cur) => {
        const error = getError.call(this, cur, this.state[cur]);

        if (error) {
            acc[cur] = error; // eslint-disable-line no-param-reassign
        }

        return acc;
    }, {});
}

export function setInputField(input, name) {
    if (input) {
        this.inputFields[name] = input;
    } else {
        delete this.inputFields[name];
    }
}

export function focusField(field) {
    if (this.inputFields.hasOwnProperty(field)) {
        // eslint-disable-next-line react/no-find-dom-node
        const input = findDOMNode(this.inputFields[field]).querySelector('input');

        if (input) {
            input.focus();
            input.click();
        }
    }
}

export function submitPassword(_revokeTokens = revokeTokens) {
    return (dispatch, getState) => {
        const {
            common: {retpath, isPDD, origin, csrf, track_id}
        } = getState();
        const {
            captcha: {key: captchaKey}
        } = this.props;
        const {captcha, currentPassword, newPassword, passwordStrength} = this.state;
        const errors = checkForErrors.call(this);

        if (Object.keys(errors).length) {
            focusField.call(this, findError(errors));
            metrics.send([METRICS_HEADER, 'Показ ошибок']);
            this.setState({errors});
            return;
        }

        if (passwordStrength.value === -1) {
            focusField.call(this, 'newPassword');
            metrics.send([METRICS_HEADER, 'Показ ошибок']);
            this.setState({showPasswordTips: true});
            return;
        }

        this.setState({loading: true});

        api.request('/profile/password/submit', {
            retpath,
            is_pdd: isPDD,
            origin,
            csrf,
            track_id,
            captcha,
            captcha_key: captchaKey,
            csrf_token: csrf,
            current_password: currentPassword,
            password: newPassword,
            lang: this.props.language,
            revoke_web_sessions: this.data.revoke_web_sessions,
            revoke_tokens: this.data.revoke_tokens,
            revoke_app_passwords: this.data.revoke_app_passwords
        })
            .done(() => {
                _revokeTokens.call(this, retpath);
            })
            .fail((res) => {
                const errs = {};
                const mappings = {
                    password: 'currentPassword'
                };
                const exceptions = {
                    'password.equals_previous': 'newPassword',
                    'password.likephonenumber': 'newPassword',
                    'password.found_in_history': 'newPassword',
                    'password.change_forbidden': 'newPassword'
                };

                res.errors.forEach(function(error) {
                    if (error === 'captcha.required') {
                        return;
                    }

                    const [fieldName, errorCode] = error.split('.');

                    let key = fieldName;

                    if (exceptions.hasOwnProperty(error)) {
                        key = exceptions[error];
                    } else if (mappings.hasOwnProperty(fieldName)) {
                        key = mappings[fieldName];
                    }

                    errs[key] = errorCode;
                });

                focusField.call(this, findError(errs));
                dispatch(reloadCaptcha());
                this.setState({
                    loading: false,
                    errors: errs,
                    captcha: ''
                });
            });
    };
}

export function revokeTokens(retpath) {
    const data = this.data;
    const tokens = !data.revoke_tokens ? data.tokens.join(',') : '';
    const passTokens = !data.revoke_app_passwords ? data.passTokens.join(',') : '';
    const otherTokens = data.revoke_other_tokens ? data.otherTokens.join(',') : '';
    const otherYandexTokens = data.revoke_other_yandex_tokens ? data.otherYandexTokens.join(',') : '';
    const onDone = ({status, failed} = {}) => {
        if (status === 'ok') {
            const count = this.state.revokers.passTokens.length;

            this.props.dispatch(setAppPasswordsNumber(count - (data.passTokens.length - failed.passwords)));
        }

        this.setState({loading: false});

        if (this.props.isAm) {
            window.location.href = this.props.closeUrl;
            return;
        }

        if (retpath) {
            location.href = retpath;
            return;
        }

        if (this.props.closeModal) {
            this.props.closeModal();
        } else {
            this.props.dispatch(push('/profile'));
        }
    };

    api.request('/profile/password/revoke', {
        lang: this.props.language,
        tokens,
        passTokens,
        otherTokens,
        otherYandexTokens
    })
        .done(onDone)
        .fail(onDone);
}

export function toggleShowPassword() {
    this.setState(({showPassword}) => ({
        showPassword: !showPassword
    }));
}

export function getPasswordStrength(password) {
    if (!password) {
        return;
    }

    this.props.dispatch((dispatch, getState) => {
        const {
            common: {track_id, uid},
            person: {login}
        } = getState();

        api.request('password', {password, track_id, uid, login}).done(
            ({validation_warnings: validationWarnings, validation_errors: validationErrors}) => {
                let passwordStrength;

                if (validationErrors) {
                    passwordStrength = {
                        value: -1,
                        code: validationErrors[0].code
                    };
                } else if (validationWarnings) {
                    passwordStrength = {
                        value: 0,
                        code: validationWarnings[0].code
                    };
                } else {
                    passwordStrength = {
                        value: 1,
                        code: 'strong'
                    };
                }

                this.passwordLineLastUpdate = Date.now();
                this.setState({passwordStrength});
            }
        );
    });
}

export function getRevokers() {
    return (dispatch, getState) => {
        const {
            common: {retpath, isPDD, track_id: trackID}
        } = getState();

        api.request('/profile/password/revokers', {
            is_pdd: isPDD,
            track_id: trackID,
            retpath,
            lang: this.props.language
        })
            .done(({revokers, state} = {}) => {
                if (!revokers || state) {
                    return this.setState({
                        revokers: {
                            error: state || 'exception.unhandled'
                        }
                    });
                }

                const def = revokers.default;

                for (const key in def) {
                    if (def.hasOwnProperty(key)) {
                        this.data[key] = def[key];
                        if (key === 'revoke_tokens') {
                            this.data.revoke_other_tokens = def[key];
                            this.data.revoke_other_yandex_tokens = def[key];
                        }
                    }
                }

                [
                    {key: 'otherTokens'},
                    {key: 'otherYandexTokens'},
                    {key: 'passTokens', dependsOn: 'revoke_app_passwords'},
                    {key: 'tokens', dependsOn: 'revoke_tokens'}
                ].forEach(({key, dependsOn}) => {
                    if (Array.isArray(revokers[key])) {
                        this.data[key] = !dependsOn || def[dependsOn] ? reduceTokens(revokers[key]) : [];
                    } else {
                        this.data[key] = [];
                        revokers[key] = []; // eslint-disable-line no-param-reassign
                    }
                });

                return this.setState({revokers});
            })
            .fail(({errors = []}) =>
                this.setState({
                    revokers: {
                        error: errors[0] || 'exception.unhandled'
                    }
                })
            );
    };
}

// eslint-disable-next-line no-useless-escape
export const checkForNumbersOrSpecialSymbols = (text) => /[0-9.,`!@#$%^&*()":;?_+=\\|\/{}\[\]<>]/.test(text);

export function onFormSubmit(event) {
    event.preventDefault();
    this.props.dispatch(submitPassword.call(this));
}

export function onFieldFocus(event) {
    const {name} = event.target;

    this.activeField = name;

    if (name === 'newPassword') {
        this.setState({showPasswordTips: true});
    } else if (this.state.showPasswordTips) {
        this.setState({showPasswordTips: false});
    }
}

export function onTextChange(event) {
    const {target = {}} = event;
    const {value, name} = target;

    if (name === 'newPassword') {
        this.getPasswordStrength(value);
    }

    if (name !== 'captcha' && !metricsSent.hasOwnProperty(name)) {
        metricsSent[name] = true;
        metrics.send([METRICS_HEADER, fieldsMetrics[name]]);
    }

    changeText.call(this, {text: value, name});
}

export function findError(errors) {
    for (let i = 0; i < fields.length; i++) {
        if (errors.hasOwnProperty(fields[i])) {
            return fields[i];
        }
    }

    return null;
}

export function setActiveError(errors) {
    if (errors === this.state.errors) {
        return;
    }

    if (Object.keys && Object.keys(errors).length === 0) {
        this.activeError = null;
        return;
    }

    if (this.activeField !== this.activeError && errors[this.activeField]) {
        this.activeError = this.activeField;
        return;
    }

    this.activeError = findError(errors);
}

export function getNamesByTokens(key, tokens) {
    return this.state.revokers[key]
        .filter(({token, title}) => title && tokens.indexOf(token) === -1)
        .map(({title}) => `«${title}»`)
        .join(', ');
}

export function reduceTokens(data) {
    return data.map(({token}) => token);
}
