import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import debounce from 'lodash/debounce';

import {Input} from '@components/Input';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';
import {Icon} from '@components/IconLego';

import ReturnToService from '@components/ReturnToService/ReturnToService.jsx';
import {Captcha} from '@components/Captcha';
import PasswordIcon from './icon.jsx';
import PasswordTip from './tip.jsx';
import PasswordMessage from './message.jsx';
import PasswordError from './error.jsx';
import Revoke from './revoke.jsx';

import * as extracted from './password.js';

export const STAFF_LINK =
    'https://doc.yandex-team.ru/help/diy/common/auth/change-password-staff.html#change-password-staff';

export default class Password extends React.PureComponent {
    constructor(props) {
        super(props);

        this.getPasswordStrength = debounce(extracted.getPasswordStrength.bind(this), 500, {
            leading: true,
            trailing: true
        });
        this.onFormSubmit = extracted.onFormSubmit.bind(this);
        this.onFieldFocus = extracted.onFieldFocus.bind(this);
        this.onTextChange = extracted.onTextChange.bind(this);
        this.onEyeClick = extracted.toggleShowPassword.bind(this);
        this.onFieldBlur = () => this.setState({showPasswordTips: false});
        this.toggleCheckList = () =>
            this.setState(({showCheckList}) => ({
                showCheckList: !showCheckList
            }));
        this.applyData = (data) => {
            this.data = data;
        };

        this.eye = `${props.staticPath}/i/icon_opened_eye.svg`;
        this.eyeClosed = `${props.staticPath}/i/icon_closed_eye.svg`;
        this.activeField = null;
        this.activeError = null;
        this.data = {
            revoke_other_tokens: false,
            revoke_other_yandex_tokens: false,
            revoke_app_passwords: false,
            revoke_tokens: false,
            revoke_web_sessions: false,
            tokens: [],
            passTokens: [],
            otherTokens: [],
            otherYandexTokens: []
        };
        this.passwordLineLastUpdate = Date.now();
        this.state = {
            showCheckList: false,
            revokers: {
                loading: true
            },
            currentPassword: '',
            newPassword: '',
            repeatPassword: '',
            captcha: '',
            errors: {},
            passwordStrength: {
                value: -1,
                code: ''
            },
            loading: false,
            showPasswordTips: false,
            showPassword: false
        };
        this.inputFields = {};
    }

    componentDidMount() {
        extracted.sendOpenMetrics();
        this.props.dispatch(extracted.getRevokers.call(this));
    }

    componentWillUpdate(_, {errors}) {
        extracted.setActiveError.call(this, errors);
    }

    getNamesByTokens(key, tokens) {
        return this.state.revokers[key]
            .filter(({token}) => !tokens.includes(token))
            .map(({title, created}) => (
                <li className='changepass__revoked' key={created}>
                    {title || i18n('Profile.devices.default-device')}
                    {created && ` (${i18n('Profile2.created')} ${created})`}
                </li>
            ));
    }

    getWarningTextStart = ({hasAnyRevoke, revokeWebSessions, hasAnyTokenRevoke, hasAnyAppPasswordRevoke} = {}) => {
        if (hasAnyRevoke) {
            if (revokeWebSessions && !hasAnyTokenRevoke && !hasAnyAppPasswordRevoke) {
                return `${i18n('Profile2.revoke.after')} ${i18n('Profile2.revoke.web_sessions')}`;
            }

            if (revokeWebSessions && hasAnyTokenRevoke && hasAnyAppPasswordRevoke) {
                return `
                    ${i18n('Profile2.revoke.after')}
                    ${i18n('Profile2.revoke.web_sessions')},
                    ${i18n('Profile2.revoke')}
                `;
            }

            if (revokeWebSessions && (hasAnyTokenRevoke || hasAnyAppPasswordRevoke)) {
                return `
                    ${i18n('Profile2.revoke.after')} 
                    ${i18n('Profile2.revoke.web_sessions')} 
                    ${i18n('common.and')} 
                    ${i18n('Profile2.revoke')}
                `;
            }

            if (!revokeWebSessions && (hasAnyTokenRevoke || hasAnyAppPasswordRevoke)) {
                return `${i18n('Profile2.revoke.after')} ${i18n('Profile2.revoke')}`;
            }
        }

        return '';
    };

    getFirstWarningText = ({
        warningTextStart,
        hasAnyTokenRevoke,
        hasAnyAppPasswordRevoke,
        tokensChanged,
        passesChanged,
        isAllowedSelect = true
    } = {}) => {
        if (isAllowedSelect) {
            if (!hasAnyTokenRevoke && !hasAnyAppPasswordRevoke) {
                return `${warningTextStart}.`;
            }

            if (hasAnyTokenRevoke && !tokensChanged && !hasAnyAppPasswordRevoke) {
                return `${warningTextStart} ${i18n('Profile2.revoke.tokens')}.`;
            }

            if (hasAnyTokenRevoke && tokensChanged) {
                return `${warningTextStart} ${i18n('Profile2.revoke.tokens')}, ${i18n('Profile2.changepass.besides')}:`;
            }

            if (hasAnyTokenRevoke && !tokensChanged && hasAnyAppPasswordRevoke && !passesChanged) {
                return `
                    ${warningTextStart}
                    ${i18n('Profile2.revoke.tokens')}
                    ${i18n('common.and')}
                    ${i18n('Profile2.revoke.app_passwords')}.
                `;
            }

            if (hasAnyTokenRevoke && !tokensChanged && hasAnyAppPasswordRevoke && passesChanged) {
                return `
                    ${warningTextStart}
                    ${i18n('Profile2.revoke.tokens')}
                    ${i18n('common.and')}
                    ${i18n('Profile2.revoke.app_passwords')}
                    ${i18n('Profile2.changepass.besides')}:
                `;
            }

            if (!hasAnyTokenRevoke && hasAnyAppPasswordRevoke && !passesChanged) {
                return `
                    ${warningTextStart}
                    ${i18n('Profile2.revoke.app_passwords')}.
                `;
            }

            if (!hasAnyTokenRevoke && hasAnyAppPasswordRevoke && passesChanged) {
                return `
                    ${warningTextStart}
                    ${i18n('Profile2.revoke.app_passwords')}
                    ${i18n('Profile2.changepass.besides')}:
                `;
            }
        } else {
            if (!hasAnyTokenRevoke && !hasAnyAppPasswordRevoke) {
                return `${warningTextStart}.`;
            }

            if (hasAnyTokenRevoke && !hasAnyAppPasswordRevoke) {
                return `${warningTextStart} ${i18n('Profile2.revoke.tokens')}.`;
            }

            if (!hasAnyTokenRevoke && hasAnyAppPasswordRevoke) {
                return `${warningTextStart} ${i18n('Profile2.revoke.app_passwords')}.`;
            }

            if (hasAnyTokenRevoke && hasAnyAppPasswordRevoke) {
                return `
                    ${warningTextStart} 
                    ${i18n('Profile2.revoke.tokens')} 
                    ${i18n('common.and')}
                    ${i18n('Profile2.revoke.app_passwords')}.
                `;
            }
        }

        return '';
    };

    getSecondWarningText = ({hasAnyTokenRevoke, hasAnyAppPasswordRevoke, tokensChanged, passesChanged} = {}) => {
        if (hasAnyTokenRevoke && tokensChanged && hasAnyAppPasswordRevoke && passesChanged) {
            return `
                ${i18n('Profile2.revoke.app_passwords')}
                ${i18n('Profile2.changepass.besides')}:
            `;
        }

        if (hasAnyTokenRevoke && tokensChanged && hasAnyAppPasswordRevoke && !passesChanged) {
            return `${i18n('Profile2.revoke.app_passwords')}.`;
        }

        return null;
    };

    renderNoteList = ({key} = {}) => {
        const {
            tokens,
            passTokens,
            revoke_other_tokens: revokeOtherTokens,
            revoke_other_yandex_tokens: revokeOtherYandexTokens
        } = this.data;

        switch (key) {
            case 'tokens':
                return (
                    <ul className='changepass__note-list' data-t='changepass:device-tokens'>
                        {this.getNamesByTokens(key, tokens)}
                        {!revokeOtherTokens && (
                            <li className='changepass__revoked' key='other_ts'>
                                {i18n('Profile2.other_services')}
                            </li>
                        )}
                        {!revokeOtherYandexTokens && (
                            <li className='changepass__revoked' key='other_yandex_ts'>
                                {i18n('Profile.devices.yandex')}
                            </li>
                        )}
                    </ul>
                );
            case 'passTokens':
                return (
                    <ul className='changepass__note-list' data-t='changepass:app-passwords'>
                        {this.getNamesByTokens('passTokens', passTokens)}
                    </ul>
                );
            default:
                return null;
        }
    };

    renderForm() {
        const {login, captcha, dispatch, language, isMobileCaptcha, isStrongPolicyOn, isIntranet} = this.props;
        const {
            errors,
            showPassword,
            loading,
            currentPassword,
            captcha: captchaValue,
            newPassword,
            repeatPassword,
            showCheckList,
            passwordStrength,
            revokers = {},
            showPasswordTips
        } = this.state;

        const {
            tokens,
            passTokens,
            revoke_web_sessions: revokeWebSessions,
            revoke_other_tokens: revokeOtherTokens,
            revoke_other_yandex_tokens: revokeOtherYandexTokens
        } = this.data;
        const isForceHidden = showPasswordTips && newPassword !== undefined && newPassword.length > 0;
        const tokensChanged =
            revokers.allow_select &&
            (tokens.length !== revokers.tokens.length || !revokeOtherTokens || !revokeOtherYandexTokens);
        const passesChanged = revokers.allow_select && passTokens.length !== revokers.passTokens.length;
        const hasAnyTokenRevoke = tokens.length > 0 || revokeOtherTokens || revokeOtherYandexTokens;
        const hasAnyAppPasswordRevoke = passTokens.length > 0;
        const hasAnyRevoke = revokeWebSessions || hasAnyAppPasswordRevoke || hasAnyTokenRevoke;
        const warningTextStart = this.getWarningTextStart({
            hasAnyRevoke,
            revokeWebSessions,
            hasAnyTokenRevoke,
            hasAnyAppPasswordRevoke
        });

        return (
            <div
                className={classnames({
                    changepass__hidden: showCheckList
                })}
            >
                {isIntranet && (
                    <div
                        className='changepass__intranet-disclaimer'
                        dangerouslySetInnerHTML={{
                            __html: i18n('Profile2.intranet.disclaimer').replace('%href%', STAFF_LINK)
                        }}
                    />
                )}
                <div className='changepass__current-password'>
                    <label className='changepass__label' aria-hidden='true' htmlFor='currentPassword'>
                        {i18n('Profile2.changepass.current_password')}
                    </label>
                    <Input
                        size='l'
                        autoComplete='current-password'
                        className={errors.currentPassword ? 'changepass__input-error' : undefined}
                        ref={(input) => extracted.setInputField.call(this, input, 'currentPassword')}
                        type='password'
                        value={currentPassword}
                        onChange={this.onTextChange}
                        onFocus={this.onFieldFocus}
                        id='currentPassword'
                        name='currentPassword'
                        autoFocus={true}
                    />
                    <PasswordError
                        error={errors.currentPassword}
                        activeError={this.activeError}
                        isForceHidden={isForceHidden}
                        fieldName='currentPassword'
                    />
                </div>
                <input type='text' className='changepass__login' autoComplete='username' value={login} />
                <div className='changepass__new-password'>
                    <label className='changepass__label' aria-hidden='true' htmlFor='newPassword'>
                        {i18n('Frontend.field_password_new')}
                    </label>
                    <Input
                        size='l'
                        autoComplete='new-password'
                        className={
                            errors.newPassword || (passwordStrength.value === -1 && newPassword.length > 0)
                                ? 'changepass__input-error'
                                : undefined
                        }
                        ref={(input) => extracted.setInputField.call(this, input, 'newPassword')}
                        type={showPassword ? 'text' : 'password'}
                        value={newPassword}
                        iconRight={
                            <PasswordIcon url={showPassword ? this.eyeClosed : this.eye} onClick={this.onEyeClick} />
                        }
                        onFocus={this.onFieldFocus}
                        onBlur={this.onFieldBlur}
                        onChange={this.onTextChange}
                        id='newPassword'
                        name='newPassword'
                    />
                    <PasswordError error={errors.newPassword} activeError={this.activeError} fieldName='newPassword' />
                    <PasswordTip
                        error={errors.newPassword}
                        activeError={this.activeError}
                        isStrongPolicyOn={isStrongPolicyOn}
                        newPassword={newPassword || ''}
                        passwordStrength={passwordStrength}
                        isVisible={!errors.newPassword && showPasswordTips}
                        passwordLineLastUpdate={this.passwordLineLastUpdate}
                    />
                </div>
                <div className='changepass__repeat-password'>
                    <label className='changepass__label' aria-hidden='true' htmlFor='repeatPassword'>
                        {i18n('Profile2.changepass.repeat_password')}
                    </label>
                    <Input
                        size='l'
                        autoComplete='new-password'
                        className={errors.repeatPassword ? 'changepass__input-error' : undefined}
                        ref={(input) => extracted.setInputField.call(this, input, 'repeatPassword')}
                        type={showPassword ? 'text' : 'password'}
                        value={repeatPassword}
                        onFocus={this.onFieldFocus}
                        onChange={this.onTextChange}
                        id='repeatPassword'
                        name='repeatPassword'
                    />
                    <PasswordError
                        error={errors.repeatPassword}
                        activeError={this.activeError}
                        isForceHidden={isForceHidden}
                        fieldName='repeatPassword'
                    />
                </div>
                <div className='changepass__captcha-block'>
                    <label className='changepass__label' htmlFor='captcha'>
                        {i18n('Profile2.changepass.captcha')}
                    </label>
                    <div className='changepass__captcha'>
                        <Captcha
                            isMobile={isMobileCaptcha}
                            lang={language}
                            introSound={captcha.introSound}
                            captchaSound={captcha.captchaSound}
                            imageUrl={captcha.imageUrl}
                            loading={captcha.loading}
                            loadingAudio={captcha.loadingAudio || false}
                            playing={captcha.playing}
                            type={captcha.type}
                            dispatch={dispatch}
                            visible={true}
                        />
                    </div>
                    <div className='changepass__captcha-input'>
                        <Input
                            size='l'
                            className={errors.captcha ? 'changepass__input-error' : undefined}
                            ref={(input) => extracted.setInputField.call(this, input, 'captcha')}
                            value={captchaValue}
                            onChange={this.onTextChange}
                            onFocus={this.onFieldFocus}
                            id='captcha'
                            name='captcha'
                        />
                        <PasswordError
                            error={errors.captcha}
                            activeError={this.activeError}
                            isForceHidden={errors.captcha === 'required'}
                            fieldName='captcha'
                        />
                    </div>
                </div>
                {(revokers.allow_select || hasAnyRevoke) && (
                    <div className='changepass__note'>
                        {revokers.allow_select ? (
                            <span data-t='changepass:select'>
                                {this.getFirstWarningText({
                                    warningTextStart,
                                    hasAnyTokenRevoke,
                                    hasAnyAppPasswordRevoke,
                                    tokensChanged,
                                    passesChanged,
                                    isAllowedSelect: true
                                })}

                                {hasAnyTokenRevoke && tokensChanged && this.renderNoteList({key: 'tokens'})}

                                {this.getSecondWarningText({
                                    hasAnyTokenRevoke,
                                    hasAnyAppPasswordRevoke,
                                    tokensChanged,
                                    passesChanged
                                })}

                                {hasAnyAppPasswordRevoke && passesChanged && this.renderNoteList({key: 'passTokens'})}

                                {!hasAnyRevoke && i18n('Profile2.changepass.no-change')}
                                {!showCheckList && (
                                    <span role='link' tabIndex='0' className='d-link' onClick={this.toggleCheckList}>
                                        <span onClick={extracted.sendToggleMetrics}> {i18n('_AUTH_.change')}</span>
                                    </span>
                                )}
                            </span>
                        ) : (
                            <span data-t='changepass:no-select'>
                                {this.getFirstWarningText({
                                    warningTextStart,
                                    hasAnyTokenRevoke,
                                    hasAnyAppPasswordRevoke,
                                    isAllowedSelect: false
                                })}
                            </span>
                        )}
                    </div>
                )}
                {errors.backend && <div className='changepass__error-box'>{i18n('Verification.errors.internal')}</div>}
                <div className='changepass__controls'>
                    {loading ? (
                        <Spin size='m' progress={true} />
                    ) : (
                        <Button view='action' type='submit' size='l' width={isMobileCaptcha ? 'max' : 'auto'}>
                            {i18n('Profile.common.continue')}
                        </Button>
                    )}
                </div>
            </div>
        );
    }

    render() {
        const revokers = this.state.revokers;
        const {isModal, retpath, isAm, closeUrl} = this.props;

        return (
            <div>
                <div
                    className={classnames('p-control-form', {
                        'changepass__no-return': isModal && !this.state.showCheckList,
                        'changepass__with-return': isModal
                    })}
                >
                    {isAm && (
                        <button
                            className='changepass__close-am-button'
                            onClick={() => {
                                window.location.href = closeUrl;
                            }}
                        >
                            <Icon size='xs' glyph='type-cross-websearch' />
                        </button>
                    )}
                    <ReturnToService
                        pathTo='/profile'
                        retpath={retpath}
                        onReturn={this.state.showCheckList ? this.toggleCheckList : undefined}
                    />
                    {revokers.error ? (
                        <PasswordMessage code={revokers.error} />
                    ) : (
                        <form className='changepass' onSubmit={this.onFormSubmit}>
                            <h2 className='section-title'>{i18n('Frontend.title.changepassword')}</h2>
                            {!revokers.loading && revokers.allow_select && this.state.showCheckList && (
                                <Revoke
                                    initialData={this.data}
                                    revokers={revokers}
                                    isModal={isModal}
                                    sendData={this.applyData}
                                    close={this.toggleCheckList}
                                />
                            )}
                            {revokers.loading ? (
                                <div className='changepass__loading'>
                                    <Spin size='l' progress={true} />
                                </div>
                            ) : (
                                this.renderForm()
                            )}
                        </form>
                    )}
                </div>
            </div>
        );
    }
}

Password.defaultProps = {
    isMobileCaptcha: false
};

Password.propTypes = {
    isIntranet: PropTypes.bool,
    login: PropTypes.string.isRequired,
    onSuccess: PropTypes.func,
    dispatch: PropTypes.func.isRequired,
    staticPath: PropTypes.string.isRequired,
    captcha: PropTypes.object.isRequired,
    isMobileCaptcha: PropTypes.bool.isRequired,
    isStrongPolicyOn: PropTypes.bool.isRequired,
    language: PropTypes.string.isRequired,
    isModal: PropTypes.func,
    retpath: PropTypes.string,
    isAm: PropTypes.bool,
    closeUrl: PropTypes.string
};
