import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import PasswordLine from './strength.jsx';
import Popup from './popup.jsx';
import Error from './error.jsx';

import {checkForNumbersAndCases} from './tip.js';

const messages = {
    tooshort: i18n('_AUTH_.password_errors_tooshort'),
    toolong: i18n('_AUTH_.password_errors_toolong'),
    tooweak: i18n('_AUTH_.password_errors_tooweak'),
    weak: i18n('_AUTH_.password_errors_weak'),
    prohibitedsymbols: i18n('_AUTH_.password_errors_prohibitedsymbols'),
    likelogin: i18n('_AUTH_.password_errors_likelogin'),
    likephonenumber: i18n('_AUTH_.password_errors_likephonenumber'),
    likeOldPassword: i18n('_AUTH_.password_errors_equals_previous'),
    foundInHistory: i18n('_AUTH_.password_errors_foundinhistory'),
    strong: i18n('_AUTH_.password_valid')
};

const tipMessages = {
    tooshort: i18n('_AUTH_.password_error_description-tooshort'),
    tooweak: i18n('_AUTH_.password_errors_tooweak_description'),
    forbidden_words: i18n('Profile.password.tips.forbidden-words'),
    weak: i18n('_AUTH_.password_errors_description-weak'),
    strong: i18n('Profile.password.tips.strong'),
    super_strong: i18n('Frontend.password_valid_description'),
    prohibitedsymbols: i18n('_AUTH_.password_errors_description-prohibited-symbols')
};

export default function PasswordTip(props) {
    const {
        error,
        activeError,
        newPassword,
        passwordStrength: {code, value},
        isVisible,
        isStrongPolicyOn,
        passwordLineLastUpdate
    } = props;
    const lineCodes = ['tooshort', 'tooweak', 'weak', 'strong'];
    const hasTipMessage = tipMessages.hasOwnProperty(code);

    if (!code) {
        return null;
    }

    if (value === -1 && !isVisible && !activeError) {
        return (
            <Error isForceShown={true} fieldName='newPassword'>
                <div className='changepass__error'>{messages[code]}</div>
            </Error>
        );
    }

    const tip =
        newPassword.length > 0 ? ( // eslint-disable-line no-extra-parens
            <div
                className={classnames('changepass__tips', {
                    changepass__tips_unacceptable: value === -1,
                    changepass__tips_weak: value === 0,
                    changepass__tips_strong: value === 1
                })}
            >
                <h3
                    className={classnames('changepass__tips-title', {
                        'changepass__tips-title_error': !hasTipMessage
                    })}
                >
                    {code === 'weak' && value === 0 ? messages.tooweak : messages[code]}
                </h3>
                {lineCodes.indexOf(code) !== -1 && (
                    <PasswordLine lastUpdate={passwordLineLastUpdate} text={newPassword} strength={value} />
                )}
                {hasTipMessage && (
                    <div className='changepass__tips-text' aria-live='polite'>
                        {(() => {
                            if (code === 'weak' && value === 0) {
                                return tipMessages.tooweak;
                            } else if (code === 'strong' && checkForNumbersAndCases(newPassword)) {
                                return tipMessages.super_strong;
                            } else if (code === 'tooshort' && isStrongPolicyOn) {
                                return tipMessages.tooshort.replace(/\d+/, '10');
                            }

                            return tipMessages[code];
                        })()}
                    </div>
                )}
            </div>
        ) : null;

    return (
        <div>
            {tip && !error && <div className='changepass__descriptor_mobile'>{tip}</div>}
            <Popup visible={isVisible}>{tip}</Popup>
        </div>
    );
}

PasswordTip.propTypes = {
    error: PropTypes.string,
    activeError: PropTypes.string,
    newPassword: PropTypes.string.isRequired,
    passwordStrength: PropTypes.object.isRequired,
    isVisible: PropTypes.bool.isRequired,
    isStrongPolicyOn: PropTypes.bool,
    passwordLineLastUpdate: PropTypes.number.isRequired
};
