import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import MordaLite from './lite.jsx';
import MordaDefault from './default.jsx';
import ScrollToService from '@blocks/morda/dashboard/components/scroll_to_service/scroll_to_service.jsx';

import {context} from '../../hocs/withContext';
import {deepEqual} from '../../utils';
import {hasExp} from '../../selectors';

const SECTION_CN_PREFIX = 'pst_';
const KEYS = ['subscriptions', 'card', 'maps', 'mail', 'key', 'clock', 'social', 'other-settings'];
const OLD_KEYS = ['key', 'clock', 'mail', 'card', 'social', 'maps', 'other-settings'];
const BLOCK_TO_KEY_MAP = {
    AuthBlock: 'key',
    Subscriptions: 'subscriptions',
    BankCards: 'card',
    MailsBlock: 'mail',
    HistoryBlock: 'clock',
    SocialBlock: 'social',
    AddressesBlock: 'maps',
    OtherSettingsBlock: 'other-settings'
};
const KEY_TO_BLOCK_MAP = Object.entries(BLOCK_TO_KEY_MAP).reduce((acc, [key, value]) => {
    acc[value] = key;
    return acc;
}, {});

export default class Morda extends React.Component {
    constructor(props) {
        super(props);

        const {
            settings: {tld, isLite, ua},
            common: {v2, isWSUser, locationCountryId}
        } = props;
        const keys = v2 ? KEYS : OLD_KEYS;
        const isPassportCardsComEnableExp = hasExp(props, 'passport-cards-com-enable');
        const isPassportGDPREnabled = hasExp(props, 'passport-gdpr-enabled');
        const isYandexIosApp =
            (ua.BrowserName === 'YandexSearch' || ua.BrowserName === 'YandexBrowser') && ua.OSFamily === 'iOS';
        const showSubscriptionsCountryIds = [
            10099, // Монголия
            168, // Армения
            167, // Азербайджан
            149, // Беларусь
            159, // Казахстан
            207, // Кыргызстан
            208, // Молдова
            209, // Таджикистан
            171, // Узбекистан
            181, // Израиль
            225 // Россия
        ];

        const hideSubscriptions = isYandexIosApp && !showSubscriptionsCountryIds.includes(locationCountryId);

        this.isKUBR = ['kz', 'ua', 'by', 'ru'].includes(tld);
        this.visibleBlocks = {
            AuthBlock: !isLite,
            HistoryBlock: !isLite,
            MailsBlock: true,
            BankCards: this.isKUBR || isPassportCardsComEnableExp,
            SocialBlock: !isWSUser && !isLite,
            AddressesBlock: this.isKUBR,
            OtherSettingsBlock: isPassportGDPREnabled || tld === 'ru',
            Subscriptions: !hideSubscriptions
        };
        this.services = keys.map((key) => this.visibleBlocks[KEY_TO_BLOCK_MAP[key]] && key).filter(Boolean);
        this.order = keys.map((key) => KEY_TO_BLOCK_MAP[key]);
    }

    getTileClassName(key) {
        return `section ${SECTION_CN_PREFIX}${key}`;
    }

    componentWillReceiveProps(nextProps) {
        if (!deepEqual(nextProps, this.props, 'common.edit')) {
            context.setItem('rerender', 'all');
        } else {
            context.setItem('rerender', nextProps.common.edit);
        }
    }

    render() {
        const {
            bnpl,
            person,
            common,
            access,
            emails,
            phones,
            social,
            devices,
            billing,
            captcha,
            security,
            settings,
            dispatch,
            addresses,
            changeAvatar,
            appPasswords,
            mordaSubscriptionCards
        } = this.props;
        const {isLite, isNewLayout} = settings;

        return (
            <div>
                {!isLite && !common.v2 && isNewLayout && (
                    <ScrollToService
                        forceListen={true}
                        services={this.services}
                        prefix={SECTION_CN_PREFIX}
                        isMobile={settings.isPhone}
                        sectionClassName='section'
                    />
                )}
                <div
                    className={classnames({
                        profile__margin: !isNewLayout,
                        profile__margin_nl: isNewLayout,
                        profile__margin_v2: common.v2 && !isNewLayout
                    })}
                >
                    <div className='profile__wrapper'>
                        <div className={isNewLayout ? undefined : `profile__inner${common.v2 ? '_v2' : ''}`}>
                            {isLite ? (
                                <MordaLite
                                    bnpl={bnpl}
                                    order={this.order}
                                    access={access}
                                    person={person}
                                    common={common}
                                    emails={emails}
                                    phones={phones}
                                    billing={billing}
                                    security={security}
                                    settings={settings}
                                    dispatch={dispatch}
                                    addresses={addresses}
                                    changeAvatar={changeAvatar}
                                    visibleBlocks={this.visibleBlocks}
                                    getTileClassName={this.getTileClassName}
                                    mordaSubscriptionCards={mordaSubscriptionCards}
                                />
                            ) : (
                                <MordaDefault
                                    bnpl={bnpl}
                                    order={this.order}
                                    person={person}
                                    common={common}
                                    access={access}
                                    emails={emails}
                                    phones={phones}
                                    social={social}
                                    devices={devices}
                                    billing={billing}
                                    captcha={captcha}
                                    security={security}
                                    settings={settings}
                                    dispatch={dispatch}
                                    isKUBR={this.isKUBR}
                                    addresses={addresses}
                                    changeAvatar={changeAvatar}
                                    appPasswords={appPasswords}
                                    visibleBlocks={this.visibleBlocks}
                                    getTileClassName={this.getTileClassName}
                                    mordaSubscriptionCards={mordaSubscriptionCards}
                                />
                            )}
                        </div>
                    </div>
                </div>
            </div>
        );
    }
}

Morda.propTypes = {
    bnpl: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    common: PropTypes.object.isRequired,
    access: PropTypes.object.isRequired,
    person: PropTypes.object.isRequired,
    emails: PropTypes.object.isRequired,
    phones: PropTypes.object.isRequired,
    social: PropTypes.object.isRequired,
    devices: PropTypes.object.isRequired,
    billing: PropTypes.object.isRequired,
    captcha: PropTypes.object.isRequired,
    settings: PropTypes.object.isRequired,
    security: PropTypes.object.isRequired,
    addresses: PropTypes.object.isRequired,
    appPasswords: PropTypes.object.isRequired,
    changeAvatar: PropTypes.object.isRequired,
    mordaSubscriptionCards: PropTypes.object.isRequired
};
