import React, {Component} from 'react';
import PropTypes from 'prop-types';
import metrics from '@blocks/metrics';
import {Modal} from '@components/Modal';
import {Spin} from '@components/Spin';
import {Checkbox} from '@components/Checkbox';
import {fetchSettings, postHelping, postRevokeDevice, postApproveDevice} from '../api';
import {AliceDeviceList} from '../AliceDeviceList';
import {AliceRevokeModal} from '../AliceRevokeModal';
import {disapproveDevice} from '../actions/actions';

export class AliceSettings extends Component {
    state = {
        isRevokeModalOpened: false,
        revokeModalDevice: null
    };
    onChangeAliceHelping = (e) => {
        const value = e.target.checked;
        const {dispatch} = this.props;

        dispatch(postHelping(value));
        metrics.send([`Новый профиль: Алиса: ${value ? 'включение' : 'снятие'} галочки "Помогать Алисе учиться"`]);
    };

    revokeOnClick = (revokeModalDevice) => {
        this.setState({isRevokeModalOpened: true, revokeModalDevice});
    };

    onRevokeAccept = () => {
        const {dispatch} = this.props;
        const {revokeModalDevice} = this.state;

        dispatch(postRevokeDevice(revokeModalDevice.id));
        this.setState({isRevokeModalOpened: false});
    };

    onRevokeDecline = () => {
        this.setState({isRevokeModalOpened: false});
    };

    approveOnClick = (device) => {
        const {dispatch} = this.props;

        dispatch(postApproveDevice(device.id));
    };
    disapproveOnClick = (device) => {
        const {dispatch} = this.props;

        dispatch(disapproveDevice(device.id));
    };

    componentDidMount() {
        const {dispatch} = this.props;

        dispatch(fetchSettings());

        metrics.send(['Новый профиль: Алиса: показ попапа настроек']);
    }

    maybeRenderDevicesBlock = () => {
        const {
            alice: {
                isLoading,
                devices: {approved = [], disapproved = []}
            }
        } = this.props;
        const showApprovedDevices = Boolean(approved.length);
        const showDissapprovedDevices = Boolean(disapproved.length);
        const showDevicesListTitle = showDissapprovedDevices || showApprovedDevices;

        if (isLoading) {
            return (
                <div className='s-alice-settings__spin'>
                    <Spin progress={true} size='l' />
                </div>
            );
        }

        return (
            <>
                {showDevicesListTitle && (
                    <>
                        <div className='s-alice-settings__devices-title'>
                            {i18n('Profile2.settings.alice.devices.title')}
                        </div>
                        <div className='s-alice-settings__devices-subtitle'>
                            {i18n('Profile2.settings.alice.devices.subtitle')}
                        </div>
                    </>
                )}

                {showDissapprovedDevices && (
                    <AliceDeviceList
                        isApproved={false}
                        devices={disapproved}
                        revokeOnClick={this.revokeOnClick}
                        approveOnClick={this.approveOnClick}
                        disapproveOnClick={this.disapproveOnClick}
                    />
                )}
                {showApprovedDevices && (
                    <AliceDeviceList
                        isApproved={true}
                        devices={approved}
                        revokeOnClick={this.revokeOnClick}
                        approveOnClick={this.approveOnClick}
                        disapproveOnClick={this.disapproveOnClick}
                    />
                )}
            </>
        );
    };

    render() {
        const {
            alice: {
                settings: {isHelpEnabled},
                isLoading
            },
            settings: {isTouch}
        } = this.props;
        const {isRevokeModalOpened, revokeModalDevice} = this.state;

        return (
            <div className='s-alice-settings' data-t='pssp:alice-settings'>
                {this.maybeRenderDevicesBlock()}

                <div className='s-alice-settings__label'>{i18n('Profile2.settings.alice.helping_label')}</div>

                <span className='s-alice-settings__checkbox' data-t='pssp:alice-settings-helping'>
                    <Checkbox
                        size='m'
                        name='alice-help'
                        checked={isHelpEnabled}
                        disabled={isLoading}
                        onChange={this.onChangeAliceHelping}
                    />
                </span>

                <div className='s-alice-settings__helping-description'>
                    <div className='s-alice-settings__helping-description_first-section'>
                        {i18n('Profile2.settings.alice.helping_description.first')}
                    </div>
                    <div className='s-alice-settings__helping-description_second-section'>
                        {i18n('Profile2.settings.alice.helping_description.second')}
                    </div>
                </div>

                <Modal
                    visible={isRevokeModalOpened}
                    contentVerticalAlign={isTouch ? 'bottom' : 'middle'}
                    dataT='alice-settings'
                >
                    {isRevokeModalOpened && (
                        <AliceRevokeModal
                            device={revokeModalDevice}
                            onRevokeAccept={this.onRevokeAccept}
                            onRevokeDecline={this.onRevokeDecline}
                        />
                    )}
                </Modal>
            </div>
        );
    }
}

AliceSettings.propTypes = {
    dispatch: PropTypes.func.isRequired,
    alice: PropTypes.object.isRequired,
    location: PropTypes.object.isRequired,
    settings: PropTypes.object.isRequired,
    devices: PropTypes.arrayOf(PropTypes.object),
    modal: PropTypes.bool
};
