import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {Link as RLink} from 'react-router-dom';
import {setCookie} from '@blocks/utils';
import passport from '@plibs/pclientjs/js/passport';
import {hasExp} from '@blocks/selectors';

import {Link} from '@components/Link';
import Modal from '@components/Wrappers/Modal.jsx';
import {Drawer} from '@components/Drawer';
import {Spin} from '@components/Spin';

import {PersonalInfoEditForm} from '@blocks/morda/personal_info/personal_info_form.jsx';
import {Avatar} from '@blocks/morda/components/avatar/avatar.jsx';
import DisplayNameEditForm from '@blocks/morda/personal_info/display_name_edit_form.jsx';

import metrics from '@blocks/metrics';

import {withUpdater} from '@blocks/hocs/withContext';
import withMethods from '@blocks/hocs/withMethods';

import * as extracted from './personal_info.js';
import {AdditionalPersonalInfo} from './AdditionalPersonalInfo';
import {FamilyAlertSwitch} from './FamilyAlertSwitch';
import {SecurityAlert} from './SecurityAlert/SecurityAlert';

const alertIconProvider = (type) => {
    return (classname) => <div className={classnames([`p-icon p-icon__${type}`, classname])} />;
};

const bnplTitles = {
    0: () => i18n('Profile2.bnpl.title.day.0'),
    1: () => i18n('Profile2.bnpl.title.day.1'),
    2: () => i18n('Profile2.bnpl.title.day.2'),
    3: () => i18n('Profile2.bnpl.title.day.3'),
    4: () => i18n('Profile2.bnpl.title.day.4'),
    5: () => i18n('Profile2.bnpl.title.day.5'),
    6: () => i18n('Profile2.bnpl.title.day.6'),
    today: () => i18n('Profile2.bnpl.title.day.today'),
    tomorrow: () => i18n('Profile2.bnpl.title.day.tomorrow')
};

const getBnplTitle = (key, money, date) => {
    let getTitle = bnplTitles[key];

    if (!getTitle) {
        getTitle = () => i18n('Profile2.bnpl.title.day.date');
    }

    return getTitle()
        .replace('{money}', money)
        .replace('{date}', date);
};

const getSecurityText = (level) => {
    switch (level) {
        case 'low':
            return i18n('Profile2.security.level.low');
        case 'middle':
            return i18n('Profile2.security.level.middle');
        case 'high':
            return i18n('Profile2.security.level.high.new');
        default:
            return i18n('Profile.security.title');
    }
};

class PersonalInfo extends React.Component {
    state = {
        showTipAlert: true,
        showBnplModal: this.props.bnpl.showWhenOpen,
        bnplIframeHeight: 320,
        availHeight: 320,
        bnplIsLoading: true
    };

    constructor(props) {
        super(props);

        this.closePersonalInfoEditForm = extracted.closePersonalInfoEditForm.bind(this);
    }

    onBnplMessage = (event) => {
        if (event.data.type === 'bnpl-resize') {
            this.setState({bnplIframeHeight: event.data.height});
        }

        if (event.data.type === 'bnpl-payment-cancelled') {
            this.closeBnplModal();
        }

        if (event.data.type === 'bnpl-loaded') {
            this.setState({bnplIsLoading: false});
        }
    };

    // eslint-disable-next-line compat/compat
    calcAvailHeight = () => this.setState({availHeight: window.screen.availHeight});

    componentDidMount() {
        const {bnpl: {url} = {}} = this.props;

        if (url) {
            window.addEventListener('message', this.onBnplMessage);
            window.addEventListener('resize', this.calcAvailHeight);
            this.calcAvailHeight();
        }
        extracted.sendGoal.call(this);
    }

    componentWillUnmount() {
        const {bnpl: {url} = {}} = this.props;

        if (url) {
            window.removeEventListener('message', this.onBnplMessage);
            window.removeEventListener('resize', this.calcAvailHeight);
        }
    }

    getLogin() {
        const {displayLogin, uid, isLiteUser} = this.props.person;

        const getLink = (content) => (
            <span className='personal-info-login__link'>
                <Link href='/auth/complete?origin=passport_profile'>{content}</Link>
            </span>
        );

        const getLogin = (content) => (
            <div
                className={classnames('personal-info-login__text', {
                    'personal-info-login__text_decorated': displayLogin
                })}
            >
                {content}
            </div>
        );

        if (isLiteUser) {
            return (
                <div className='personal-info-login'>
                    {getLogin(displayLogin)}
                    {getLink(i18n('Profile.create.yandex-email'))}
                </div>
            );
        }

        if (!displayLogin) {
            return (
                <div className='personal-info-login'>
                    {getLogin(`ID: ${uid}`)}
                    {getLink(i18n('Profile.create.login'))}
                </div>
            );
        }

        return <div className='personal-info-login'>{getLogin(displayLogin)}</div>;
    }

    renderSecurityAlert(link) {
        const {
            hasPhone,
            is2faEnabled,
            isSms2faEnabled,
            securityLevel,
            settings: {isTouch}
        } = this.props;

        const has2fa = isSms2faEnabled || is2faEnabled;
        const highSecurity = securityLevel === 'high';

        if (highSecurity && (!hasPhone || !has2fa)) {
            return;
        }

        return (
            <SecurityAlert
                className={classnames('personal-info__alert', {
                    'personal-info__alert_order_last': highSecurity
                })}
                title={getSecurityText(securityLevel)}
                action={highSecurity ? '' : i18n('Profile2.security.increase')}
                href={highSecurity ? undefined : link}
                icon={alertIconProvider(`exp-${securityLevel}`)}
                view={securityLevel === 'low' ? 'warning' : 'default'}
                isTouch={isTouch}
            />
        );
    }

    onClosePhoneAlert = () => {
        this.setState({showTipAlert: false});
        setCookie('security_alert_phone', 7);
    };

    onClose2faAlert = () => {
        this.setState({showTipAlert: false});
        setCookie('security_alert_2fa', 7);
    };

    render2faAlert() {
        const {
            settings: {isTouch}
        } = this.props;

        return (
            <SecurityAlert
                className='personal-info__alert'
                title={i18n('Profile2.security-tip.2fa.title')}
                action={i18n('Profile2.security-tip.2fa.action')}
                href={'/profile/access/2fa?origin=passport_profile&utm_source=id&utm_medium=alert'}
                icon={alertIconProvider('2fa')}
                isTouch={isTouch}
                hasClose={true}
                onClose={this.onClose2faAlert}
            />
        );
    }

    renderAddPhoneAlert() {
        const {
            settings: {isLite, isTouch}
        } = this.props;

        return (
            <SecurityAlert
                className='personal-info__alert'
                title={i18n('Profile2.security-tip.phone.title')}
                action={i18n('Profile2.security-tip.phone.action')}
                href={`/profile/phones?origin=passport_profile${
                    isLite ? '&lite=1' : ''
                }&utm_source=id&utm_medium=alert`}
                icon={alertIconProvider('phone')}
                isTouch={isTouch}
                hasClose={true}
                onClose={this.onClosePhoneAlert}
            />
        );
    }

    renderTipAlert() {
        const {hasPhone, is2faEnabled, isSms2faEnabled, isClosedPhoneAlert, isClosed2faAlert} = this.props;

        if (!hasPhone && !isClosedPhoneAlert) {
            return this.renderAddPhoneAlert();
        }

        if (!(is2faEnabled || isSms2faEnabled) && !isClosed2faAlert) {
            return this.render2faAlert();
        }
    }

    renderSecurityAlerts(link, hasTipExp) {
        return (
            <>
                {this.renderSecurityAlert(link)}
                {hasTipExp && this.state.showTipAlert && this.renderTipAlert()}
                {this.props.bnpl.hasNextPayment && this.renderBnplAlert()}
            </>
        );
    }

    closeBnplModal = () => this.setState({showBnplModal: false});
    openBnplModal = () => this.setState({showBnplModal: true});

    renderBnplModal() {
        const {settings: {isTouch} = {}, bnpl: {url} = {}} = this.props;
        const {showBnplModal, bnplIframeHeight, availHeight, bnplIsLoading} = this.state;
        const Component = isTouch ? Drawer : Modal;

        return (
            <Component
                className={`personal-info__bnpl-${isTouch ? 'drawer' : 'modal'}`}
                visible={showBnplModal}
                view={isTouch ? 'default' : undefined}
                onClose={this.closeBnplModal}
            >
                {bnplIsLoading && (
                    <div className='personal-info__bnpl-loader'>
                        <Spin progress={true} size='l' />
                    </div>
                )}
                <iframe
                    height={Math.min(bnplIframeHeight, availHeight - 86)}
                    className={classnames('personal-info__bnpl-iframe', {
                        'personal-info__bnpl-iframe_visible': !bnplIsLoading
                    })}
                    src={url}
                />
            </Component>
        );
    }

    renderBnplAlert() {
        const {
            bnpl,
            settings: {isTouch}
        } = this.props;

        return (
            <SecurityAlert
                className='personal-info__alert'
                title={getBnplTitle(bnpl.titleKey, `\n${bnpl.nextAmount} ₽`, bnpl.formatedDate)}
                action={i18n('Profile2.bnpl.title.left').replace('{money}', `${bnpl.amountLeft} ₽`)}
                icon={alertIconProvider('split')}
                isTouch={isTouch}
                onClick={this.openBnplModal}
            />
        );
    }

    render() {
        const {settings, common, dispatch, changeAvatar, person, securityLevel, methods, bnpl} = this.props;
        const {isPhone, isTouch} = settings;
        const {isWSUser, edit, v2 = false} = common;
        const {avatarId, firstname, lastname, login} = person;
        const noAvatar = avatarId === '0/0-0';
        const hasSecurity = typeof securityLevel !== 'undefined' && securityLevel !== 'undefined' && !isPhone;
        const hasTipExp = this.props.common.experiments.flags.includes('passport-security-tips-exp');
        const showNewSecurityAlert = typeof securityLevel !== 'undefined' && securityLevel !== 'undefined';
        const passportGetCookie = passport && passport.util && passport.util.getCookie;
        const availabeFamilyAlerts =
            !(passportGetCookie && passport.util.getCookie('profile_alert_plus')) ||
            !(passportGetCookie && passport.util.getCookie('profile_alert_pay'));
        const showAlerts = (hasSecurity && !showNewSecurityAlert) || showNewSecurityAlert || availabeFamilyAlerts;
        const showFamilyAlert = hasExp({common}, 'profile-family-alert') && availabeFamilyAlerts;
        const SECURITY_LINK = `/profile/security?origin=passport_profile${
            settings.isLite ? '&lite=1' : ''
        }&utm_source=id&utm_medium=alert`;

        return (
            <div className={classnames('personal', {'personal-layout__new': showAlerts})}>
                {showAlerts && (
                    <div
                        className={classnames('personal-info__alerts-gorizontal', {
                            'personal-info__alerts-gorizontal_reverse': securityLevel === 'high',
                            'personal-info__alerts-gorizontal_align_right': !hasTipExp
                        })}
                    >
                        {showFamilyAlert && <FamilyAlertSwitch />}

                        {this.renderSecurityAlerts(SECURITY_LINK, hasTipExp)}
                    </div>
                )}
                <Avatar
                    avatarId={avatarId}
                    login={login}
                    paths={settings.avatar}
                    isPhone={isTouch}
                    dispatch={dispatch}
                    edit={edit}
                    changeAvatar={changeAvatar}
                    settings={settings}
                >
                    {noAvatar &&
                        (isTouch ? (
                            <RLink className='personal-info__add-avatar d-link' to='/profile/avatars'>
                                {i18n('Profile2.add.photo')}
                            </RLink>
                        ) : (
                            <div className='personal-info__add-avatar d-link'>{i18n('Profile2.add.photo')}</div>
                        ))}
                </Avatar>
                <div
                    className={classnames('personal-info', {
                        'personal-info__no-avatar': noAvatar
                    })}
                >
                    <div className='personal-info__main-block'>
                        <div
                            className={classnames('personal-info-name font-weight_bold', {
                                'personal-info-name_max': !hasSecurity
                            })}
                        >
                            <div className='personal-info__first'>{firstname}</div>
                            <div className='personal-info__last'>{lastname}</div>
                        </div>

                        {this.getLogin()}

                        <AdditionalPersonalInfo methods={methods} expLayout={showAlerts} />
                    </div>

                    {showAlerts && (
                        <div className='personal-info__alerts-vertical'>
                            {hasSecurity && !showNewSecurityAlert && (
                                <div className={classnames(['personal__security', v2 && 'personal__security_v2'])}>
                                    <div className={`personal__security-icon p-icon p-icon__${securityLevel}`} />
                                    <div className='personal__security-aside'>
                                        <div className='personal__security-level'>{getSecurityText(securityLevel)}</div>
                                        {securityLevel !== 'high' && (
                                            <div onClick={methods.security}>
                                                <Link
                                                    className={classnames({
                                                        'personal__security-low': securityLevel === 'low'
                                                    })}
                                                    href={SECURITY_LINK}
                                                >
                                                    {i18n('Profile.security.check.level')}
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </div>
                            )}

                            {showFamilyAlert && <FamilyAlertSwitch />}

                            {this.renderSecurityAlerts(SECURITY_LINK, hasTipExp)}
                        </div>
                    )}
                </div>
                <div>
                    {!isWSUser && (
                        <Modal visible={edit === 'personal-info'} zIndexGroupLevel={-0.5} dataT='personal-info'>
                            {edit === 'personal-info' && (
                                <PersonalInfoEditForm
                                    modal={true}
                                    edit={edit}
                                    {...this.props}
                                    close={this.closePersonalInfoEditForm}
                                />
                            )}
                        </Modal>
                    )}
                    <Modal visible={edit === 'display-name'} dataT='display-name'>
                        <DisplayNameEditForm
                            modal={true}
                            settings={settings}
                            common={common}
                            person={person}
                            dispatch={dispatch}
                            close={this.closePersonalInfoEditForm}
                        />
                    </Modal>
                    {bnpl.url && this.renderBnplModal()}
                </div>
            </div>
        );
    }
}

PersonalInfo.propTypes = {
    bnpl: PropTypes.object,
    person: PropTypes.object.isRequired,
    common: PropTypes.object.isRequired,
    settings: PropTypes.object.isRequired,
    changeAvatar: PropTypes.object.isRequired,
    dispatch: PropTypes.func.isRequired,
    securityLevel: PropTypes.string,
    methods: PropTypes.object.isRequired,
    hasPhone: PropTypes.bool,
    is2faEnabled: PropTypes.bool,
    isSms2faEnabled: PropTypes.bool,
    isClosedPhoneAlert: PropTypes.bool,
    isClosed2faAlert: PropTypes.bool
};

export default withUpdater(
    withMethods(PersonalInfo, metrics.send.bind(metrics), {
        personalInfo: [extracted.METRICS_HEADER, 'Открыть форму'],
        birthday: [extracted.METRICS_HEADER, 'Открыть форму (указать дату рождения)'],
        security: ['Безопасность аккаунта', 'Защитить аккаунт']
    }),
    ['display-name', 'personal-info', 'edit-avatar'],
    'common.edit'
);
