import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import Arrow from '../arrow/arrow.jsx';

import * as extracted from './scroll.js';

export default class Scroll extends React.Component {
    state = {scroll: 0};
    scrolled = 0;
    constructor(props) {
        super(props);

        this.scroll = extracted.onScroll.bind(this);

        extracted.construct.call(this, props);
    }

    componentDidMount() {
        extracted.listenForResize.call(this);
        extracted.updateAsyncIfNotFixed.call(this);
    }

    componentWillUnmount() {
        clearTimeout(this.timer);
        if (this.destroyListener) {
            this.destroyListener();
        }
    }

    componentWillReceiveProps({activeIndex}) {
        const isDiffNumber = typeof activeIndex === 'number' && this.props.activeIndex !== activeIndex;
        const count = this.cardsInfo.count;
        const max = this.props.items.length - count;

        if (isDiffNumber) {
            this.scroll(undefined, Math.min(Math.floor(activeIndex / count) * count, max));
        }
    }

    shouldComponentUpdate(nextProps, nextState) {
        return (
            nextProps.forceUpdate ||
            nextState.scroll !== this.state.scroll ||
            nextProps.items.length !== this.props.items.length
        );
    }

    setScrollRef = (node) => {
        this.scrollRef = {current: node};
    };

    render() {
        const {isTouch, items, fixed, marginRight, flex} = this.props;
        const scroll = !isTouch ? this.state.scroll : 0;
        const rightHidden =
            !this.cardsInfo ||
            this.scrolled === items.length - this.cardsInfo.count ||
            this.cardsInfo.width * items.length <
                (this.scrollRef && this.scrollRef.current ? this.scrollRef.current.clientWidth : 0);

        return (
            <div
                className={classnames('scroll', {
                    'scroll-touch': isTouch
                })}
                ref={this.setScrollRef}
            >
                <div className={`scroll__wrapper scroll__wrapper-${isTouch ? 'touch' : 'desktop'}`}>
                    <div
                        className={classnames('scroll__inner', {
                            'scroll__inner-touch': isTouch,
                            'scroll__inner-desktop': !isTouch,
                            scroll__flex: flex
                        })}
                        style={{transform: `translateX(-${scroll || 0}${fixed ? 'px' : '%'})`}}
                    >
                        <div
                            className={classnames('scroll__content', {
                                'scroll__content-touch': isTouch,
                                'scroll__content-desktop': !isTouch,
                                scroll__flex: flex
                            })}
                        >
                            {items.map((item, ndx) => (
                                <div
                                    className='scroll__item'
                                    style={{
                                        marginRight: `${marginRight}${fixed ? 'px' : '%'}`,
                                        flex:
                                            flex && this.cardsInfo
                                                ? `0 0 ${this.cardsInfo.width - marginRight}px`
                                                : undefined
                                    }}
                                    key={ndx}
                                >
                                    {item}
                                </div>
                            ))}
                        </div>
                    </div>
                </div>
                {!isTouch && (
                    <div>
                        <Arrow hidden={scroll === 0} direction='left' onClick={this.scroll} />
                        <Arrow hidden={rightHidden} onClick={this.scroll} />
                    </div>
                )}
            </div>
        );
    }
}

Scroll.propTypes = {
    activeIndex: PropTypes.number,
    forceUpdate: PropTypes.bool,
    isTouch: PropTypes.bool.isRequired,
    items: PropTypes.array.isRequired,
    metrics: PropTypes.arrayOf(PropTypes.string),
    sizes: PropTypes.arrayOf(
        PropTypes.shape({
            maxWidth: PropTypes.number,
            cardCount: PropTypes.number,
            cardWidth: PropTypes.number
        })
    ).isRequired,
    fixed: PropTypes.bool,
    flex: PropTypes.bool,
    marginRight: PropTypes.number.isRequired
};
