import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import metrika from '@blocks/metrics';
import {CARD_STATUS, CARD_TYPES} from './constants';

const b = cn('SubscriptionCard');

const DEFAULT_TITLES = {
    [CARD_TYPES.plus]: i18n('Profile2.subscriptions.title.plus'),
    [CARD_TYPES.music]: i18n('Profile2.subscriptions.title.music'),
    [CARD_TYPES.mail]: i18n('Profile2.subscriptions.title.mail.v2'),
    [CARD_TYPES.disk]: i18n('Profile2.subscriptions.title.disk'),
    [CARD_TYPES.practicum]: i18n('Profile2.subscriptions.title.practicum')
};
const DEFAULT_GENDER = {
    [CARD_TYPES.plus]: 'm',
    [CARD_TYPES.music]: 'f',
    [CARD_TYPES.mail]: 'f',
    [CARD_TYPES.disk]: 'm',
    [CARD_TYPES.practicum]: 'm'
};

const DESCRIPTIONS = {
    [CARD_TYPES.plus]: i18n('Profile2.subscriptions.description.plus'),
    [CARD_TYPES.music]: i18n('Profile2.subscriptions.description.music'),
    [CARD_TYPES.mail]: i18n('Profile2.subscriptions.description.mail.v2'),
    [CARD_TYPES.disk]: i18n('Profile2.subscriptions.description.disk'),
    [CARD_TYPES.practicum]: i18n('Profile2.subscriptions.description.practicum')
};

const STATUS_TEXTS = {
    [CARD_STATUS.loading]: i18n('Profile2.subscriptions.status.loading'),
    [CARD_STATUS.active]: i18n('Profile2.subscriptions.status.active'),
    [`${CARD_STATUS.active}_m`]: i18n('Profile2.subscriptions.status.active'),
    [`${CARD_STATUS.active}_f`]: i18n('Profile2.subscriptions.status.active.female'),
    [CARD_STATUS.offer]: i18n('Profile2.subscriptions.status.offer'),
    [CARD_STATUS.error]: i18n('Profile2.subscriptions.status.error')
};

const METRIKA_CLICK_EVENT_BY_STATUS = {
    [CARD_STATUS.active]: 'Управлять',
    [CARD_STATUS.offer]: 'Подключить',
    [CARD_STATUS.error]: 'Проверить',
    [CARD_STATUS.loading]: null
};

const METRIKA_SEND_STATUSES = [CARD_STATUS.active, CARD_STATUS.offer, CARD_STATUS.error];

export default class SubscriptionCard extends React.PureComponent {
    componentDidMount() {
        const status = this.props.status;

        if (METRIKA_SEND_STATUSES.includes(status)) {
            metrika.send(['Подписки', 'Показ карточки подписки', this.props.type]);
        }
    }

    componentDidUpdate(prevProps) {
        const status = this.props.status;

        if (METRIKA_SEND_STATUSES.includes(status) && !METRIKA_SEND_STATUSES.includes(prevProps.status)) {
            metrika.send(['Подписки', 'Показ карточки подписки', this.props.type]);
        }
    }

    getTitle() {
        const {title, type} = this.props;

        return title || DEFAULT_TITLES[type] || type;
    }

    getStatusText() {
        const {type, status, statusGender} = this.props;
        const statusGenderToUse = statusGender || DEFAULT_GENDER[type];

        return STATUS_TEXTS[`${status}_${statusGenderToUse}`] || STATUS_TEXTS[status] || status;
    }

    getDescription() {
        const {type} = this.props;

        return DESCRIPTIONS[type] || '';
    }

    getLink = () => {
        const {type, status, tld, locationCountryId} = this.props;

        switch (type) {
            case 'music':
            case 'plus':
                if ([CARD_STATUS.offer].includes(status)) {
                    return `https://plus.yandex.${tld}?from=yandexid`;
                }
                if ([CARD_STATUS.active, CARD_STATUS.error].includes(status)) {
                    return `https://plus.yandex.${tld}/my?from=yandexid`;
                }
                break;
            case 'mail':
                if ([CARD_STATUS.offer, CARD_STATUS.active, CARD_STATUS.error].includes(status)) {
                    return 'https://mail360.yandex.ru?from=yandexid';
                }
                break;
            case 'disk':
                if ([CARD_STATUS.offer, CARD_STATUS.active, CARD_STATUS.error].includes(status)) {
                    return `https://disk.yandex.${tld}/tuning?from=yandexid`;
                }
                break;
            case 'practicum': {
                const campaign = `partners_passport${locationCountryId === 225 ? '' : '_SNG'}`;

                return `https://praktikum.yandex.ru?utm_source=partners&utm_medium=passport&utm_campaign=${campaign}`;
            }
        }

        return null;
    };

    onClick = () => {
        const {type, status} = this.props;
        const link = this.getLink();

        if (link) {
            const metrikaEvent = METRIKA_CLICK_EVENT_BY_STATUS[status];

            if (metrikaEvent) {
                metrika.send(['Подписки', 'Клик на карточку подписки', `${type}.${metrikaEvent}`]);
            }
            window.open(link, '_blank');
        }
    };

    render() {
        const {type, status, isTouch} = this.props;

        const statusText = this.getStatusText();

        return (
            <div className={b({[type]: true})} onClick={this.onClick}>
                <div className={b('image', {[type]: true, [status]: true, isTouch: isTouch})} />
                <div className={b('contentWrap')}>
                    <div className={b('textWrap')}>
                        <div className={b('title', {[status]: true})}>{this.getTitle()}</div>
                        <div className={b('description')}>{this.getDescription()}</div>
                    </div>
                    <div className={b('statusText', {wrap: true})}>
                        <span className={b('statusText', {[status]: true})}>{statusText}</span>
                    </div>
                </div>
                {status === CARD_STATUS.active && <div className={b('ok')} />}
            </div>
        );
    }
}

SubscriptionCard.propTypes = {
    tld: PropTypes.string.isRequired,
    isTouch: PropTypes.bool.isRequired,
    type: PropTypes.oneOf(Object.keys(CARD_TYPES)).isRequired,
    status: PropTypes.oneOf(Object.keys(CARD_STATUS)).isRequired,
    locationCountryId: PropTypes.number,
    title: PropTypes.string,
    statusGender: PropTypes.string
};
