import React from 'react';
import PropTypes from 'prop-types';
import {getSubscriptionCards} from './index';
import SubscriptionCard from './subscription_card';
import Scroll from '@blocks/morda/new/scroll/scroll.jsx';
import {Section} from '@blocks/morda/new/section/section.jsx';
import metrika from '@blocks/metrics';
import {CARD_STATUS, CARD_TYPES, CARD_SORT_VALUE} from './constants';

const CARD_LOADING_STATE = [
    {
        type: CARD_TYPES.plus,
        status: CARD_STATUS.loading
    },
    {
        type: CARD_TYPES.mail,
        status: CARD_STATUS.loading
    }
];
const CARD_ERROR_STATE = [
    {
        type: CARD_TYPES.plus,
        status: CARD_STATUS.error
    },
    {
        type: CARD_TYPES.mail,
        status: CARD_STATUS.offer
    }
];

const sortSubscriptions = (subscriptions) =>
    [...subscriptions].sort((subscription, other) => CARD_SORT_VALUE[other.type] - CARD_SORT_VALUE[subscription.type]);

const getSortedSubscriptions = (subscriptions) => {
    const [activeSubs, offerSubs, otherSubs] = [[], [], []];

    subscriptions.forEach((subcription) => {
        if (subcription.status === CARD_STATUS.active) {
            activeSubs.push(subcription);
        } else if (subcription.status === CARD_STATUS.offer) {
            offerSubs.push(subcription);
        } else {
            otherSubs.push(subcription);
        }
    });

    return [...sortSubscriptions(activeSubs), ...sortSubscriptions(offerSubs), ...sortSubscriptions(otherSubs)];
};

export default class SubscriptionsBlock extends React.PureComponent {
    componentDidMount() {
        this.props.dispatch(getSubscriptionCards());
    }

    componentDidUpdate(oldProps) {
        const subscriptionCards = this.props.subscriptionCards;

        if (subscriptionCards !== oldProps.subscriptionCards && !subscriptionCards.plusProgress) {
            metrika.send(['Подписки', 'Показ блока']);
        }
    }

    render() {
        const {
            settings: {tld, isTouch} = {},
            common: {locationCountryId} = {},
            subscriptionCards: {plusProgress, subscriptions, plusErrors} = {}
        } = this.props;

        let subsToDisplay;

        if (plusProgress) {
            subsToDisplay = CARD_LOADING_STATE;
        } else if (Array.isArray(plusErrors) && plusErrors.length > 0) {
            subsToDisplay = CARD_ERROR_STATE;
        } else if (!subscriptions || subscriptions.length === 0) {
            return null;
        } else {
            subsToDisplay = getSortedSubscriptions(subscriptions);
        }

        const keyToResetScroll = subsToDisplay.reduce((acc, sub) => acc + sub.type + sub.status, '');

        const scrollItems = subsToDisplay.map((subscription) => (
            <SubscriptionCard
                key={subscription.type + subscription.status}
                tld={tld}
                locationCountryId={locationCountryId}
                isTouch={isTouch}
                status={subscription.status}
                type={subscription.type}
                title={subscription.title}
                statusGender={subscription.statusGender}
            />
        ));

        return (
            <Section className='p-subscriptions' icon='subscriptions' title={i18n('Profile2.titles.subscriptions')}>
                <Scroll
                    key={keyToResetScroll}
                    isTouch={isTouch}
                    items={scrollItems}
                    fixed={true}
                    sizes={[
                        {
                            maxWidth: 0,
                            cardWidth: 280
                        }
                    ]}
                    marginRight={8}
                />
            </Section>
        );
    }
}

SubscriptionsBlock.propTypes = {
    settings: PropTypes.object,
    common: PropTypes.object,
    dispatch: PropTypes.func,
    subscriptionCards: PropTypes.shape({
        plusProgress: PropTypes.bool,
        plusErrors: PropTypes.array,
        subscriptions: PropTypes.arrayOf(
            PropTypes.shape({
                type: PropTypes.string.isRequired,
                status: PropTypes.string.isRequired,
                title: PropTypes.string,
                statusGender: PropTypes.string
            })
        )
    })
};
